/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.tc5.session;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;

import javax.management.AttributeNotFoundException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.transaction.TransactionManager;

import org.apache.catalina.Context;
import org.jboss.aspects.patterns.observable.Observer;
import org.jboss.aspects.patterns.observable.Subject;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.aop.CachedType;
import org.jboss.cache.aop.TreeCacheAopMBean;
import org.jboss.cache.aop.TreeCacheAop;
import org.jboss.cache.transaction.BatchModeTransactionManager;
import org.jboss.invocation.MarshalledValue;
import org.jboss.invocation.MarshalledValueInputStream;
import org.jboss.invocation.MarshalledValueOutputStream;
import org.jboss.logging.Logger;
import org.jboss.mx.util.MBeanProxyExt;

/**
 * A wrapper class to JBossCache. This is currently needed to handle various operations such as
 * <ul>
 * <li>Using MarshalledValue to replace Serializable used inside different web app class loader context.</li>
 * <li>Stripping out any id string after ".". This is to handle the JK failover properly with
 * Tomcat JvmRoute.</li>
 * <li>Cache exception retry.</li>
 * <li>Helper APIS.</li>
 * </ul>
 */
public class JBossCacheService
{
   private TreeCacheAop proxy_;
   private ObjectName cacheServiceName_;
   protected static Logger log_ = Logger.getLogger(JBossCacheService.class);
   public static final String SESSION = "JSESSION";
   public static final String ATTRIBUTE = "ATTRIBUTE";
   public static final String KEY = "ATRR_KEY";
   // Needed for cache invalidation
   static final String VERSION_KEY = "VERSION";
   static final String FQN_DELIMITER = "/";
   // name of webapp's virtual host(JBAS-2194). 
   // Idea is host_name + web_app_path + session id is a unique combo.
   private String hostName_;
   // web app path (JBAS-1367 and JBAS-2194). 
   // Idea is host_name + web_app_path + session id is a unique combo.
   private String webAppPath_;
   private TransactionManager tm;

   // Class loader for this web app.
   private ClassLoader tcl_;
   private JBossCacheManager manager_;
   private CacheListener cacheListener_;
   private JBossCacheWrapper cacheWrapper_;
   
   // Do we have to marshall attributes ourself or can we let 
   // the TreeCache do it?
   private boolean useTreeCacheMarshalling_ = false;
   
   private MBeanServer jmxServer_ = null;
   
   public JBossCacheService(String treeCacheObjectName, MBeanServer server) throws ClusteringNotSupportedException
   {
      // Find JBossCacheService
      try
      {
         jmxServer_ = server;
         
         cacheServiceName_ = new ObjectName(treeCacheObjectName);
         // Create Proxy-Object for this service
         proxy_ = (TreeCacheAop) ((TreeCacheAopMBean) MBeanProxyExt.create(TreeCacheAopMBean.class,
                 cacheServiceName_)).getInstance();
         if (proxy_ == null)
         {
            throw new RuntimeException("JBossCacheService: locate null TomcatCacheMbean");
         }

         cacheWrapper_ = new JBossCacheWrapper(proxy_);
         
         // Determine whether we use marshalling via a call on the
         // JMX server rather than the proxy.  This allows interoperability
         // with JBossCache 1.2.3 and earlier
         //useTreeCacheMarshalling_ = proxy_.getUseMarshalling(); 
         try
         {
            Boolean utcm = (Boolean) jmxServer_.getAttribute(cacheServiceName_, "UseMarshalling");
            useTreeCacheMarshalling_ = utcm.booleanValue();
         }
         catch (AttributeNotFoundException ignore) 
         {
            log_.info("TreeCache marshalling API unavailable; will cache marshalled objects");
         }
      }
      catch (Throwable e)
      {
         String str = cacheServiceName_ + " service to Tomcat clustering not found";
         log_.error(str);
         throw new ClusteringNotSupportedException(str);
      }
   }

   public void start(ClassLoader tcl, JBossCacheManager manager)
   {
      tcl_ = tcl;
      manager_ = manager;
      
      Context webapp = (Context) manager_.getContainer();
      String path = webapp.getName();
      if( path.length() == 0 || path.equals("/")) {
         // If this is root.
         webAppPath_ = "ROOT";
      } else if ( path.startsWith("/") ) {
         webAppPath_ = path.substring(1);
      } else {
         webAppPath_ = path;
      }
      log_.debug("Old and new web app path are: " +path + ", " +webAppPath_);
      
      String host = webapp.getParent().getName();
      if( host == null || host.length() == 0) {
         hostName_ = "localhost";
      }else {
         hostName_ = host;
      }
      log_.debug("Old and new virtual host name are: " + host + ", " + hostName_);
      

      // Construct the fqn
      Object[] objs = new Object[]{SESSION, hostName_, webAppPath_};
      Fqn pathFqn = new Fqn( objs );
      
      cacheListener_ = new CacheListener(cacheWrapper_, manager_, pathFqn);
      proxy_.addTreeCacheListener(cacheListener_);

      // register the tcl and bring over the state for the webapp
      String fqnStr = pathFqn.toString();
      try {
         if(useTreeCacheMarshalling_)
         {
            log_.debug("UseMarshalling is true. We will register the fqn: " +
                        fqnStr + " with class loader" +tcl_ +
                        " and activate the webapp's Region");
            // Don't make 1.2.4 API calls through the proxy but rather through
            // the JMX server.  This allows binary compatibility with 1.2.3.
            // useTreeCacheMarshalling_ will be false if we are using 1.2.3
            // so these calls won't happen
            //proxy_.registerClassLoader(fqnStr, tcl_);
            //proxy_.activateRegion(fqnStr);
            Object[] args = { fqnStr, tcl_ };
            String[] types = { "java.lang.String", "java.lang.ClassLoader" };
            jmxServer_.invoke(cacheServiceName_, "registerClassLoader", args, types);
            Object[] args2  = { fqnStr };
            String[] types2 = { "java.lang.String" };
            jmxServer_.invoke(cacheServiceName_, "activateRegion", args2, types2);
         }
      } catch (Exception ex)
      {
         throw new RuntimeException("Can't register class loader", ex);
      }

      // We require the cache tm to be BatchModeTransactionManager now.
      tm = proxy_.getTransactionManager();
      if(useTreeCacheMarshalling_ &&  !(tm instanceof BatchModeTransactionManager) )
      {
         throw new RuntimeException("JBossCacheService.start(): JBossCacheAop transaction manager is not type BatchModeTransactionManager." +
                 " Please check the tc5-cluster-service.xml TransactionManagerClassLookup field.");
      }
   }

   public void stop()
   {
      proxy_.removeTreeCacheListener(cacheListener_);

      // Construct the fqn
      Object[] objs = new Object[]{SESSION, hostName_, webAppPath_};
      Fqn pathFqn = new Fqn( objs );

      String fqnStr = pathFqn.toString();
      if(useTreeCacheMarshalling_)
      {
            log_.debug("UseMarshalling is true. We will inactivate the fqn: " +
                       fqnStr + " and un-register its classloader");
            
         try {
            // Don't make 1.2.4 API calls through the proxy but rather through
            // the JMX server.  This allows binary compatibility with 1.2.3.
            // useTreeCacheMarshalling_ will be false if we are using 1.2.3
            // so these calls won't happen
            //proxy_.inactivateRegion(fqnStr);
            //proxy_.unregisterClassLoader(fqnStr);
            Object[] args = { fqnStr };
            String[] types = { "java.lang.String" };
            jmxServer_.invoke(cacheServiceName_, "inactivateRegion", args, types);
            jmxServer_.invoke(cacheServiceName_, "unregisterClassLoader", args, types);           
         }
         catch (Exception e) 
         {
            log_.error("During inactivation of webapp region " + fqnStr + 
                        " or un-registration of its class loader", e);
         }
      }

      // remove session data
      cacheWrapper_.evict(pathFqn);
   }

   /**
    * Get specfically the BatchModeTransactionManager.
    */
   public TransactionManager getTransactionManager()
   {
      return tm;
   }
   
   
   /**
    * Gets whether TreeCache-based marshalling is available
    */
   public boolean isMarshallingAvailable()
   {
      return useTreeCacheMarshalling_;
   }

   /**
    * Find session ids for the whole Manager instance. Note that this also depends on the web app
    * path setting.
    *
    * @return Empty list if not found.
    */
   public List findSessionIDs()
   {
      List ids = new ArrayList();
      try {
         // Construct the fqn
         Object[] objs = new Object[]{SESSION, hostName_, webAppPath_};
         Fqn path = new Fqn( objs );
         // locate children under each web app path
         Set names = proxy_.getChildrenNames(path);

         if( names == null ) return ids;
         for(Iterator it = names.iterator(); it.hasNext();) {
            Object id = it.next();
            if(id==null) continue;
            ids.add(id);
            if(log_.isTraceEnabled()) {
               log_.trace("Retrieving through web app path with fqn: " +path + " and session id: " +id);
            }
         }
      } catch (CacheException e) {
         throw new RuntimeException("JBossCacheService: exception occurred in cache getChildrenNames ... ", e);
      }
      return ids;
   }

   /**
    * Loads any serialized data in the cache into the given session
    * using its <code>readExternal</code> method.
    *
    * @return the session passed as <code>toLoad</code>, or
    *         <code>null</code> if the cache had no data stored
    *         under the given session id.
    */
   public ClusteredSession loadSession(String realId, ClusteredSession toLoad)
   {
      Fqn fqn = getSessionFqn(realId);
      byte[] sessionData = (byte[]) cacheWrapper_.get(fqn, realId);
      
      if (sessionData == null) {
         // Requested session is no longer in the cache; return null
         return null;
      }
      
      boolean firstLoad = (toLoad.getVersion() == 0);
      
      // Swap in/out the webapp classloader so we can deserialize
      // attributes whose classes are only available to the webapp
      ClassLoader prevTCL = Thread.currentThread().getContextClassLoader();
      Thread.currentThread().setContextClassLoader(tcl_);
      try
      {
         ByteArrayInputStream bais = new ByteArrayInputStream(sessionData);
         // Use MarshalledValueInputStream instead of superclass ObjectInputStream
         // or else there are problems finding classes with scoped loaders
         MarshalledValueInputStream input = new MarshalledValueInputStream(bais);
         toLoad.readExternal(input);
         input.close();
      }
      catch (Exception e)
      {
         log_.error("loadSession(): id: " + realId + "exception occurred during serialization: " +e);
         return null;
      }
      finally {
         Thread.currentThread().setContextClassLoader(prevTCL);
      }
      
      // The internal version of the serialized session may be less than the
      // real one due to not replicating metadata.  If our listener hasn't 
      // been keeping the outdatedVersion of the session up to date because
      // the session has never been loaded into the JBCManager cache, we 
      // need to fix the version
      if (firstLoad)
      {         
         Integer ver = (Integer) cacheWrapper_.get(fqn, VERSION_KEY);
         if (ver != null)
            toLoad.setVersion(ver.intValue());
      }
      
      return toLoad;
   }

   public void putSession(String realId, ClusteredSession session)
   {
      Fqn fqn = getSessionFqn(realId);
      
      if (session.getReplicateSessionBody())
      {
         Map map = new HashMap();
         map.put(realId, externalizeSession(session));
         // Put in (VERSION_KEY, version) after the real put for cache invalidation
         map.put(VERSION_KEY, new Integer(session.getVersion()));
         cacheWrapper_.put(fqn, map);
      }
      else
      {
         // Invalidate the remote caches
         cacheWrapper_.put(fqn, VERSION_KEY, new Integer(session.getVersion()));
      }
   }

   public void removeSession(String realId)
   {
      Fqn fqn = getSessionFqn(realId);
      if (log_.isDebugEnabled())
      {
         log_.debug("Remove session from distributed store. Fqn: " + fqn);
      }
      //Object obj = getUnMarshalledValue(cacheWrapper_.remove(fqn, realId));
      cacheWrapper_.remove(fqn, realId); 
      // This needs to go after object removal to support correct cache invalidation.
//      _remove(fqn, VERSION_KEY);
      // Let just remove the whole thing (including the fqn)
      cacheWrapper_.remove(fqn);
      //return obj;
   }

   public void removeSessionLocal(String realId)
   {
      Fqn fqn = getSessionFqn(realId);
      if (log_.isDebugEnabled())
      {
         log_.debug("Remove session from my own distributed store only. Fqn: " + fqn);
      }
      cacheWrapper_.evict(fqn);
   }

   public boolean exists(String realId)
   {
      Fqn fqn = getSessionFqn(realId);
      return proxy_.exists(fqn);
   }

   public Object getAttribute(String realId, String key)
   {
      Fqn fqn = getAttributeFqn(realId);
      return getUnMarshalledValue(cacheWrapper_.get(fqn, key));
   }

   public Object putAttribute(String realId, String key, Object value)
   {
      Fqn fqn = getAttributeFqn(realId);
      return cacheWrapper_.put(fqn, key, getMarshalledValue(value));
   }

   public void putAttribute(String realId, Map map)
   {
      Fqn fqn = getAttributeFqn(realId);
      Set set = map.keySet();
      Iterator it = set.iterator();
      while (it.hasNext())
      {
         String key = (String) it.next();
         cacheWrapper_.put(fqn, key, getMarshalledValue(map.get(key)));
      }
   }

   public void removeAttributes(String realId)
   {
      Fqn fqn = getAttributeFqn(realId);
      cacheWrapper_.remove(fqn);
   }

   public Object removeAttribute(String realId, String key)
   {
      Fqn fqn = getAttributeFqn(realId);
      if (log_.isTraceEnabled())
      {
         log_.trace("Remove attribute from distributed store. Fqn: " + fqn + " key: " + key);
      }
      return getUnMarshalledValue(cacheWrapper_.remove(fqn, key));
   }

   public void removeAttributesLocal(String realId)
   {
      Fqn fqn = getAttributeFqn(realId);
      if (log_.isDebugEnabled())
      {
         log_.debug("Remove attributes from my own distributed store only. Fqn: " + fqn);
      }
      cacheWrapper_.evict(fqn);
   }

   /**
    * Obtain the keys associated with this fqn. Note that it is not the fqn children.
    *
    */
   public Set getAttributeKeys(String realId)
   {
      Set keys = null;
      Fqn fqn = getAttributeFqn(realId);
      try
      {
         keys = proxy_.getKeys(fqn);
      }
      catch (CacheException e)
      {
         log_.error("getAttributeKeys(): Exception getting keys for session " + realId, e);
      }
      
      return keys;
   }

   /**
    * Return all attributes associated with this session id.
    *
    * @param realId the session id with any jvmRoute removed
    * @return the attributes, or any empty Map if none are found.
    */
   public Map getAttributes(String realId)
   {
      if (realId == null || realId.length() == 0) return new HashMap();
      
      Map map = new HashMap();
      Set set = getAttributeKeys(realId);
      if(set != null)
      {
         for (Iterator it = set.iterator(); it.hasNext();)
         {
            String key = (String) it.next();
            Object value = getAttribute(realId, key);
            map.put(key, value);
         }
      }
      return map;
   }

   /**
    * Gets the ids of all sessions in the underlying cache.
    *
    * @return Set containing all of the session ids of sessions in the cache
    *         (with any jvmRoute removed) or <code>null</code> if there
    *         are no sessions in the cache.
    */
   public Set getSessionIds()
   {
      Set result = null;
      try
      {
         result = proxy_.getChildrenNames(getWebappFqn());
      }
      catch (CacheException e)
      {
         log_.error("getSessionIds(): Exception getting session ids for " +
                    "webapp " + webAppPath_, e);
      }

      return result;
   }

   /**
    * store the pojo instance in the cache. Note that this is for the aop cache.
    * THe pojo needs to be "aspectized".
    * 
    * @param realId the session id with any jvmRoute removed
    * @param key    the attribute key
    * @param pojo
    */
   public Object setPojo(String realId, String key, Object pojo)
   {
      // TODO This actually redundant here -- its already in
      // FieldBasedClusteredSession
      if( !Util.checkPojoType(pojo) )
      {
         throw new RuntimeException("setPojo: pojo is not an instance of Advised. Need to declare it in aop: "
                 +pojo.toString());
      }

      if(log_.isTraceEnabled())
      {
         log_.trace("setPojo(): session id: " + realId + " key: " + key + 
                    " object: " + pojo.toString());
      }
      // Construct the fqn.
      Fqn fqn = getFieldFqn(realId, key);
      try {
         // Ignore any cache notifications that our own work generates 
         LocalSessionActivity.startLocalActivity(realId);
         return proxy_.putObject(fqn, pojo);
      } catch (CacheException e) {
         throw new RuntimeException("JBossCacheService: exception occurred in cache setPojo ... ", e);
      }
      finally {
         LocalSessionActivity.finishLocalActivity();
      }
   }

   /**
    * Remove pojo from the underlying cache store.
    * @param realId the session id with any jvmRoute removed
    * @param key    the attribute key
    * @return pojo that just removed. Null if there none.
    */
   public Object removePojo(String realId, String key)
   {
      if(log_.isTraceEnabled())
      {
         log_.trace("removePojo(): session id: " +realId + " key: " +key);
      }
      // Construct the fqn.
      Fqn fqn = getFieldFqn(realId, key);
      try {
         // Ignore any cache notifications that our own work generates 
         LocalSessionActivity.startLocalActivity(realId);
         return proxy_.removeObject(fqn);
      } catch (CacheException e) {
         throw new RuntimeException("JBossCacheService: exception occurred in cache removePojo ... ", e);
      }
      finally {
         LocalSessionActivity.finishLocalActivity();
      }
   }

   /**
    * Remove all the pojos from the underlying cache store locally 
    * without replication.
    * 
    * @param realId the session id with any jvmRoute removed
    */
   public void removePojosLocal(String realId)
   {
      if(log_.isDebugEnabled())
      {
         log_.debug("removePojoLocal(): session id: " +realId);
      }
      // Construct the fqn.
      Fqn fqn = getAttributeFqn(realId);
      try {
         // Ignore any cache notifications that our own work generates 
         LocalSessionActivity.startLocalActivity(realId);
         cacheWrapper_.evictSubtree(fqn);
      }
      finally {
         LocalSessionActivity.finishLocalActivity();
      }
   }

   /**
    * Remove all the pojos from the underlying cache store locally 
    * without replication.
    * 
    * @param realId the session id with any jvmRoute removed
    */
   public void removePojoLocal(String realId, String key)
   {
      if(log_.isTraceEnabled())
      {
         log_.trace("removePojoLocal(): session id: " + realId + " key: " +key);
      }
      // Construct the fqn.
      Fqn fqn = getFieldFqn(realId, key);
      try {
         // Ignore any cache notifications that our own work generates 
         LocalSessionActivity.startLocalActivity(realId);
         cacheWrapper_.evictSubtree(fqn);
      }
      finally {
         LocalSessionActivity.finishLocalActivity();
      }
   }
   
   public Set getPojoKeys(String realId)
   {
      Set keys = null;
      Fqn fqn = getAttributeFqn(realId);
      try
      {
         keys = proxy_.getChildrenNames(fqn);
      }
      catch (CacheException e)
      {
         log_.error("getPojoKeys(): Exception getting keys for session " + realId, e);
      }
      
      return keys;      
   }
   

   /**
    *
    * @param realId the session id with any jvmRoute removed
    * @param key    the attribute key
    * @return Pojo that is associated with the attribute
    */
   public Object getPojo(String realId, String key)
   {
      if(log_.isTraceEnabled())
      {
         log_.trace("getPojo(): session id: " +realId + " key: " +key);
      }
      // Construct the fqn.
      Fqn fqn = getFieldFqn(realId, key);

      // Swap in/out the webapp classloader so we can deserialize
      // attributes whose classes are only available to the webapp
//      ClassLoader prevTCL = Thread.currentThread().getContextClassLoader();
//      Thread.currentThread().setContextClassLoader(tcl_);

      try {
         return proxy_.getObject(fqn);
      } catch (CacheException e) {
         throw new RuntimeException("JBossCacheService: exception occurred in cache getPojo ... ", e);
      } finally
      {
//         Thread.currentThread().setContextClassLoader(prevTCL);
      }
   }

   /**
    * Recursively adds session as observer to the pojo graph. Assumes the 
    * whole object graph has Subject "introduction" declared. If a portion
    * of the graph isn't a Subject, the recursion does not continue below
    * that part of the graph.
    *  
    * @param session  the session
    * @param pojo     the pojo.  Can be <code>null</code>.
    */
   public void addObserver(Observer session, Object pojo)
   {
      if ( pojo instanceof Collection )
      {
         Collection col = (Collection)pojo;
         for (Iterator i = col.iterator(); i.hasNext();) {
            Object obj = i.next();
            // If not a managed pojo, will return anyway
            addObserver(session, obj);
         }

         return;
      } else if (pojo instanceof Map)
      {
         Map map = (Map)pojo;
         for (Iterator i = map.keySet().iterator(); i.hasNext();) {
            Object key = i.next();
            Object value = map.get(key);

            // Walk thru key and value
            addObserver(session, key);
            addObserver(session, value);
         }

         return;
      }

      // BRIAN 3/14 changed this from checking Advised to checking Subject
      // since that is what we cast to below
      if(! (pojo instanceof Subject) )
      {
         return;  // No need to add observer since it is primitive.
      }

      Subject subject = (Subject)pojo;
      subject.addObserver(session);
      if(log_.isTraceEnabled())
      {
         log_.trace("addObserver(): session: " +session + " pojo name: " +pojo.getClass().getName());
      }
      // Traverse recursively
      CachedType type = proxy_.getCachedType(pojo.getClass());
      for (Iterator i = type.getFields().iterator(); i.hasNext();) {
         Field field = (Field) i.next();
         Object value = null;
         try {
            value=field.get(pojo);
         }
         catch(IllegalAccessException e) {
            throw new RuntimeException("field access failed", e);
         }
         CachedType fieldType = proxy_.getCachedType(field.getType());
         if (fieldType.isImmediate()) {
            continue;
         } else {
            addObserver(session, value);
         }
      }
   }

   /**
    * Recursively removes session as observer to the pojo graph. Assumes the 
    * whole object graph has Subject "introduction" declared. If a portion
    * of the graph isn't a Subject, the recursion does not continue below
    * that part of the graph.
    *  
    * @param session  the session
    * @param pojo the pojo to stop observing.  Can be <code>null</code>.
    */
   public void removeObserver(Observer session, Object pojo)
   {
      // BRIAN 3/14 changed this from checking Advised to checking Subject
      // since that is what we cast to below
      if(! (pojo instanceof Subject) )
      {
         return;  // No need to add observer since it is primitive.
      }

      Subject subject = (Subject)pojo;
      subject.removeObserver(session);
      if(log_.isTraceEnabled())
      {
         log_.trace("removeObserver(): session: " +session + " pojo name: " +pojo.getClass().getName());
      }
      // Traverse recursively
      CachedType type = proxy_.getCachedType(pojo.getClass());
      for (Iterator i = type.getFields().iterator(); i.hasNext();) {
         Field field = (Field) i.next();
         Object value = null;
         try {
            value=field.get(pojo);
         }
         catch(IllegalAccessException e) {
            throw new RuntimeException("field access failed", e);
         }
         CachedType fieldType = proxy_.getCachedType(field.getType());
         if (fieldType.isImmediate()) {
            continue;
         } else {
            removeObserver(session, value);
         }
      }
   }

   private Fqn getFieldFqn(String id, String key)
   {
      // /SESSION/id/ATTR/key
      // Guard against string with delimiter.
      List list = new ArrayList();
      list.add(SESSION);
      list.add(hostName_);
      list.add(webAppPath_);
      list.add(id);
      list.add(ATTRIBUTE);
      breakKeys(key, list);
      return new Fqn(list);
   }

   private void breakKeys(String key, List list)
   {
      StringTokenizer token = new StringTokenizer(key, FQN_DELIMITER);
      while(token.hasMoreTokens())
      {
         list.add(token.nextToken());
      }
   }

   private Fqn getWebappFqn()
   {
      // /SESSION/hostname/webAppPath
      Object[] objs = new Object[]{SESSION, hostName_, webAppPath_};
      return new Fqn(objs);
   }
   
   private Fqn getSessionFqn(String id)
   {
      // /SESSION/hostname/webAppPath/id
      Object[] objs = new Object[]{SESSION, hostName_, webAppPath_, id};
      return new Fqn(objs);
   }

   private Fqn getAttributeFqn(String id)
   {
      // /SESSION/hostName/webAppPath/id/ATTR
      Object[] objs = new Object[]{SESSION, hostName_, webAppPath_, id, ATTRIBUTE};
      return new Fqn(objs);
   }

   private Object getMarshalledValue(Object value)
   {
      // JBAS-2920.  For now, continue using MarshalledValue, as 
      // it allows lazy deserialization of the attribute on remote nodes
      // For Branch_4_0 this is what we have to do anyway for backwards
      // compatibility. For HEAD we'll follow suit for now.
      // TODO consider only using MV for complex objects (i.e. not primitives)
      // and Strings longer than X.
      
//      if (useTreeCacheMarshalling_)
//      {
//         return value;
//      }
//      else
//      {
         try
         {
            MarshalledValue mv = new MarshalledValue(value);
            if (log_.isTraceEnabled())
            {
               log_.trace("marshalled object to size " + mv.size() + " bytes");
            }
            return mv;
         }
         catch (IOException e)
         {
            log_.error("IOException occurred marshalling value ", e);
            return null;
         }
//      }
   }

   private Object getUnMarshalledValue(Object mv)
   {
      // JBAS-2920.  For now, continue using MarshalledValue, as 
      // it allows lazy deserialization of the attribute on remote nodes
      // For Branch_4_0 this is what we have to do anyway for backwards
      // compatibility. For HEAD we'll follow suit for now.
//      if (useTreeCacheMarshalling_)
//      {
//         return mv;
//      }
//      else
//      {
         if (mv == null) return null;
         // Swap in/out the tcl for this web app. Needed only for un marshalling.
         ClassLoader prevTCL = Thread.currentThread().getContextClassLoader();
         Thread.currentThread().setContextClassLoader(tcl_);
         try
         {
            return ((MarshalledValue) mv).get();
         }
         catch (IOException e)
         {
            log_.error("IOException occurred unmarshalling value ", e);
            return null;
         }
         catch (ClassNotFoundException e)
         {
            log_.error("ClassNotFoundException occurred unmarshalling value ", e);
            return null;
         }
         finally
         {
            Thread.currentThread().setContextClassLoader(prevTCL);
         }
//      }
   }

   private byte[] externalizeSession(ClusteredSession session)
   {      
      try
      {
         // Write the contents of session to a byte array and store that
         ByteArrayOutputStream baos = new ByteArrayOutputStream();
         // Use MarshalledValueOutputStream instead of superclass ObjectOutputStream
         // or else there are problems finding classes with scoped loaders
         MarshalledValueOutputStream oos = new MarshalledValueOutputStream(baos);
         session.writeExternal(oos);
         oos.close(); // flushes bytes to baos
         
         byte[] bytes = baos.toByteArray();
         
         if (log_.isTraceEnabled())
         {
            log_.trace("marshalled object to size " + bytes.length + " bytes");
         }

         return bytes;
      }
      catch (Exception e)
      {
         log_.error("externalizeSession(): exception occurred externalizing session " + session, e);
         return null;
      }
      
   }

}
