/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ejb3.injection;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import javax.annotation.Resource;
import javax.annotation.Resources;
import javax.ejb.EJBContext;
import javax.ejb.TimerService;
import javax.naming.Context;
import javax.transaction.UserTransaction;
import org.jboss.ejb3.Container;
import org.jboss.ejb3.EJBContainer;
import org.jboss.ejb3.dd.Injectable;
import org.jboss.ejb3.dd.MessageDestinationRef;
import org.jboss.ejb3.dd.ResourceEnvRef;
import org.jboss.ejb3.dd.ResourceRef;
import org.jboss.ejb3.interceptor.InterceptorInjector;
import org.jboss.logging.Logger;

/**
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @version $Revision$
 */
public class ResourceHandler
{
   private static final Logger log = Logger.getLogger(ResourceHandler.class);


   private static void loadXmlResourceRefs(EJBContainer container, Collection<ResourceRef> refs, Class clazz, HashMap<AccessibleObject, Injector> injectors)
   {
      for (ResourceRef envRef : refs)
      {
         if (envRef.getMappedName() == null || envRef.getMappedName().equals(""))
            throw new RuntimeException("mapped-name is required for " + envRef.getResRefName() + " of EJB " + container.getEjbName());
         else
            container.addEncLinkRefEntry("env/" + envRef.getResRefName(), envRef.getMappedName());

         if (envRef.getInjectionTarget() != null)
         {
            // todo, get injection target class
            AccessibleObject ao = InjectionUtil.findInjectionTarget(clazz, envRef.getInjectionTarget());
            if (ao instanceof Field)
            {
               injectors.put(ao, new JndiFieldInjector((Field) ao, "env/" + envRef.getResRefName(), container.getEnc()));
            }
            else
            {
               injectors.put(ao, new JndiMethodInjector((java.lang.reflect.Method) ao, "env/" + envRef.getResRefName(), container.getEnc()));
            }
         }
      }
   }

   private static void loadXmlResourceEnvRefs(EJBContainer container, Collection<ResourceEnvRef> refs, Class clazz, HashMap<AccessibleObject, Injector> injectors)
   {
      for (ResourceEnvRef envRef : refs)
      {
         if (envRef.getMappedName() == null || envRef.getMappedName().equals(""))
            log.warn("mapped-name is required for " + envRef.getResRefName() + " of EJB " + container.getEjbName());
         else
            container.addEncLinkRefEntry("env/" + envRef.getResRefName(), envRef.getMappedName());

         if (envRef.getInjectionTarget() != null)
         {
            // todo, get injection target class
            AccessibleObject ao = InjectionUtil.findInjectionTarget(clazz, envRef.getInjectionTarget());
            if (ao instanceof Field)
            {
               injectors.put(ao, new JndiFieldInjector((Field) ao, "env/" + envRef.getResRefName(), container.getEnc()));
            }
            else
            {
               injectors.put(ao, new JndiMethodInjector((java.lang.reflect.Method) ao, "env/" + envRef.getResRefName(), container.getEnc()));
            }
         }
      }
   }
   
   private static void loadXmlMessageDestinationRefs(EJBContainer container, Collection<MessageDestinationRef> refs, Class clazz, HashMap<AccessibleObject, Injector> injectors)
   {
      for (MessageDestinationRef envRef : refs)
      {
         if (envRef.getMappedName() == null || envRef.getMappedName().equals(""))
            log.warn("mapped-name is required for " + envRef.getMessageDestinationRefName() + " of EJB " + container.getEjbName());
         else
            container.addEncLinkRefEntry("env/" + envRef.getMessageDestinationRefName(), envRef.getMappedName());

         if (envRef.getInjectionTarget() != null)
         {
            // todo, get injection target class
            AccessibleObject ao = InjectionUtil.findInjectionTarget(clazz, envRef.getInjectionTarget());
            if (ao instanceof Field)
            {
               injectors.put(ao, new JndiFieldInjector((Field) ao, "env/" + envRef.getMessageDestinationRefName(), container.getEnc()));
            }
            else
            {
               injectors.put(ao, new JndiMethodInjector((java.lang.reflect.Method) ao, "env/" + envRef.getMessageDestinationRefName(), container.getEnc()));
            }
         }
      }
   }

   private static void loadClassRefs(EJBContainer container, Class clazz, boolean isContainer) throws Exception
   {
      Resources resources = (Resources) InjectionUtil.getAnnotation(Resources.class, container, clazz, isContainer);
      if (resources == null) return;
      for (Resource ref : resources.value())
      {
         String encName = ref.name();
         if (encName == null || encName.equals(""))
         {
            throw new RuntimeException("JBoss requires name() for class level @Resource");
         }
         String mappedName = ref.mappedName();
         if (mappedName == null || mappedName.equals(""))
         {
            if (!container.getEncLinkRefEntries().containsKey(encName))
            {
               throw new RuntimeException("You did not specify a @Resource.mappedName() on " + clazz.getName() + " and there is no binding for that enc name in XML");
            }
            continue;
         }

         if (!container.hasEncEntry(encName))
         {
            container.addEncLinkRefEntry(encName, mappedName);
         }
      }
   }


   private static void loadMethodInjectors(HashSet<String> methods, EJBContainer container, Class clazz, HashMap<AccessibleObject, Injector> injectors, Context ctx, boolean isContainer)
   {
      if (clazz == null || clazz.equals(Object.class))
      {
         return;
      }
      for (Method method : clazz.getDeclaredMethods())
      {
         Resource ref = (Resource)InjectionUtil.getAnnotation(Resource.class, container, method, isContainer);
         if (ref != null)
         {
            if (!Modifier.isPrivate(method.getModifiers()))
            {
               if (methods.contains(method.getName())) continue;
               else methods.add(method.getName());
            }
            method.setAccessible(true);

            if (injectors.containsKey(method)) continue;

            if (!method.getName().startsWith("set")) throw new RuntimeException("@Resource can only be used with a set method: " + method);
            if (method.getParameterTypes().length != 1) throw new RuntimeException("@Resource can only be used with a set method of one parameter: " + method);
            Class type = method.getParameterTypes()[0];
            if (!ref.type().equals(Object.class))
            {
               type = ref.type();
            }
            if (type.equals(UserTransaction.class))
            {
               injectors.put(method, new UserTransactionMethodInjector(method, container));
            }
            else if (type.equals(TimerService.class))
            {
               injectors.put(method, new TimerServiceMethodInjector(method, container));
            }
            else if (EJBContext.class.isAssignableFrom(type))
            {
               injectors.put(method, new EJBContextMethodInjector(method));
            }
            else if (type.equals(String.class)
                     || type.equals(Character.class)
                     || type.equals(Byte.class)
                     || type.equals(Short.class)
                     || type.equals(Integer.class)
                     || type.equals(Long.class)
                     || type.equals(Boolean.class)
                     || type.equals(Double.class)
                     || type.equals(Float.class)
                     || type.isPrimitive()
                    )
            {
               String encName = ref.name();
               if (encName == null || encName.equals(""))
               {
                  encName = InjectionUtil.getEncName(method);
               }
               else
               {
                  encName = "env/" + encName;
               }

               if (container.hasEnvEntry(encName.substring(4)))
               {
                  injectors.put(method, new JndiMethodInjector(method, encName, ctx));
               }
            }
            else
            {
               String encName = ref.name();
               if (encName == null || encName.equals(""))
               {
                  encName = InjectionUtil.getEncName(method);
               }
               else
               {
                  encName = "env/" + encName;
               }

               if (!injectors.containsKey(method))
               {
                  injectors.put(method, new JndiMethodInjector(method, encName, ctx));
               }

               String mappedName = ref.mappedName();
               if (mappedName == null || mappedName.equals(""))
               {
                  if (!container.getEncLinkRefEntries().containsKey(encName))
                  {
                     throw new RuntimeException("You did not specify a @Resource.mappedName() on " + method + " and there is no binding for that enc name in XML");
                  }
                  continue;
               }

               if (!container.hasEncEntry(encName))
               {
                  container.addEncLinkRefEntry(encName, mappedName);
               }
            }
         }
      }
      if (clazz != null && !clazz.equals(Object.class))
      {
         loadMethodInjectors(methods, container, clazz.getSuperclass(), injectors, ctx, isContainer);
      }
   }


   private static void loadFieldInjectors(EJBContainer container, Class clazz, HashMap<AccessibleObject, Injector> injectors, Context ctx, boolean isContainer)
   {
      if (clazz != null && !clazz.equals(Object.class))
      {
         loadFieldInjectors(container, clazz.getSuperclass(), injectors, ctx, isContainer);
      }
      for (Field field : clazz.getDeclaredFields())
      {
         Resource ref = (Resource)InjectionUtil.getAnnotation(Resource.class, container, field, isContainer);
         if (ref != null)
         {
            field.setAccessible(true);

            if (injectors.containsKey(field)) continue;

            Class type = field.getType();
            if (!ref.type().equals(Object.class))
            {
               type = ref.type();
            }
            if (type.equals(UserTransaction.class))
            {
               if (!isContainer)
               {
                  log.debug("Skipping UserTransaction for interceptor " + clazz);
               }
               else
               {
                  injectors.put(field, new UserTransactionFieldInjector(field, container));
               }
            }
            else if (type.equals(TimerService.class))
            {
               if (!isContainer)
               {
                  log.debug("Skipping TimerService for interceptor " + clazz);
               }
               else
               {
                  injectors.put(field, new TimerServiceFieldInjector(field, container));
               }
            }
            else if (EJBContext.class.isAssignableFrom(type))
            {
               injectors.put(field, new EJBContextFieldInjector(field));
            }
            else if (type.equals(String.class)
                     || type.equals(Character.class)
                     || type.equals(Byte.class)
                     || type.equals(Short.class)
                     || type.equals(Integer.class)
                     || type.equals(Long.class)
                     || type.equals(Boolean.class)
                     || type.equals(Double.class)
                     || type.equals(Float.class)
                     || type.isPrimitive()
                    )
            {
               String encName = ref.name();
               if (encName == null || encName.equals(""))
               {
                  encName = InjectionUtil.getEncName(field);
               }
               else
               {
                  encName = "env/" + encName;
               }
                
               if (container.hasEnvEntry(encName.substring(4)))
               {
                  injectors.put(field, new JndiFieldInjector(field, encName, ctx));
               }
            }
            else
            {
               String encName = ref.name();
               if (encName == null || encName.equals(""))
               {
                  encName = InjectionUtil.getEncName(field);
               }
               else
               {
                  encName = "env/" + encName;
               }

               if (!injectors.containsKey(field))
               {
                  injectors.put(field, new JndiFieldInjector(field, encName, ctx));
               }

               String mappedName = ref.mappedName();
               if (mappedName == null || mappedName.equals(""))
               {
                  if (!container.getEncLinkRefEntries().containsKey(encName))
                  {
                     throw new RuntimeException("You did not specify a @Resource.mappedName() on " + field + " and there is no binding for that enc name in XML");
                  }
                  continue;
               }
            

               if (!container.hasEncEntry(encName))
               {
                  container.addEncLinkRefEntry(encName, mappedName);
               }
            }
         }
      }
   }

   public static void loadInjectors(Container container) throws Exception
   {
      Class clazz = container.getBeanClass();
      EJBContainer ejb = (EJBContainer)container;
      loadInjectors(clazz, ejb.getXml(), ejb, ejb.getEncInjections(), true);
   }
   
   public static void loadInjectors(InterceptorInjector injector)throws Exception
   {
      loadInjectors(injector.getClazz(), injector.getXml(), (EJBContainer)injector.getContainer(), injector.getEncInjections(), false);
   }
   
   private static void loadInjectors(Class clazz, Injectable xml, EJBContainer ejb, HashMap<AccessibleObject, Injector> injectors, boolean isContainer) throws Exception
   {
      Context resourceCtx = ejb.getEnc();
      if (xml != null)
      {
         loadXmlResourceRefs(ejb, xml.getResourceRefs(), clazz, injectors);
         loadXmlResourceEnvRefs(ejb, xml.getResourceEnvRefs(), clazz, injectors);
         loadXmlMessageDestinationRefs(ejb, xml.getMessageDestinationRefs(), clazz, injectors);
      }
      loadClassRefs(ejb, clazz, isContainer);
      HashSet<String> visitedMethods = new HashSet<String>();
      loadMethodInjectors(visitedMethods, ejb, clazz, injectors, resourceCtx, isContainer);
      loadFieldInjectors(ejb, clazz, injectors, resourceCtx, isContainer);
   }
}
