/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.ejb3;

import org.jboss.annotation.ejb.PoolClass;
import org.jboss.aop.AspectManager;
import org.jboss.aop.ClassContainer;
import org.jboss.aop.advice.Interceptor;
import org.jboss.aop.annotation.AnnotationElement;
import org.jboss.aop.joinpoint.ConstructorInvocation;
import org.jboss.ejb3.dd.AssemblyDescriptor;
import org.jboss.ejb3.dd.EnterpriseBean;
import org.jboss.ejb3.dd.PersistenceContextRef;
import org.jboss.ejb3.injection.*;
import org.jboss.ejb3.interceptor.InterceptorInfo;
import org.jboss.ejb3.interceptor.InterceptorInfoRepository;
import org.jboss.ejb3.interceptor.InterceptorInjector;
import org.jboss.ejb3.interceptor.LifecycleInterceptorHandler;
import org.jboss.ejb3.security.JaccHelper;
import org.jboss.ejb3.tx.UserTransactionImpl;
import org.jboss.logging.Logger;
import org.jboss.naming.Util;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.ejb.Local;
import javax.ejb.Remote;
import javax.ejb.Timeout;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.LinkRef;
import javax.naming.NamingException;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.reflect.*;
import java.util.*;

/**
 * Comment
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @version $Revision$
 */
public abstract class EJBContainer extends ClassContainer implements Container
{

   private static final Logger log = Logger.getLogger(EJBContainer.class);

   protected Pool pool;

   protected String ejbName;

   protected ObjectName objectName;

   protected int defaultConstructorIndex;

   protected String beanClassName;

   protected ClassLoader classloader;

   protected Injector[] injectors;

   protected Context enc;
   protected Context encEnv;

   protected Class beanContextClass;

   //protected SessionCallbackHandler callbackHandler;
   protected LifecycleInterceptorHandler callbackHandler;

   protected Hashtable initialContextProperties;

   protected HashMap envEntries = new HashMap();

   protected HashMap<String, String> encLinkRefEntries = new HashMap<String, String>();

   protected HashMap<String, String> ejbRefDependencies = new HashMap<String, String>();
   protected HashMap<String, String> puEncXmlEntries = new HashMap<String, String>();
   protected HashMap<String, PersistenceContextRef> pcEncXmlEntries = new HashMap<String, PersistenceContextRef>();

   protected EnterpriseBean xml;
   protected AssemblyDescriptor assembly;

   protected HashMap<AccessibleObject, Injector> encInjections = new HashMap<AccessibleObject, Injector>();

   protected InterceptorInfoRepository interceptorRepository;

   protected List<InterceptorInfo> classInterceptors = new ArrayList<InterceptorInfo>();

   protected LinkedHashSet<InterceptorInfo> applicableInterceptors;

   private HashMap<Class, InterceptorInjector> interceptorInjectors;

   private Ejb3Deployment deployment;

   private DependencyPolicy dependencyPolicy;

   private String jaccContextId;
   protected HashMap extendedPCs = new HashMap();
   
   protected HashMap invokedMethod = new HashMap();


   /**
    *
    * @param name Advisor name
    * @param manager Domain to get interceptor bindings from
    * @param cl the EJB's classloader
    * @param beanClassName
    * @param ejbName
    * @param ctxProperties
    * @param interceptorRepository
    * @param deployment
    */

   public EJBContainer(String name, AspectManager manager, ClassLoader cl,
                       String beanClassName, String ejbName, Hashtable ctxProperties,
                       InterceptorInfoRepository interceptorRepository, Ejb3Deployment deployment)
   {
      super(name, manager);
      this.xml = xml;
      this.deployment = deployment;
      this.beanClassName = beanClassName;
      this.classloader = cl;
      try
      {
         clazz = classloader.loadClass(beanClassName);
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException(e);
      }
      this.ejbName = ejbName;
      String on = Ejb3Module.BASE_EJB3_JMX_NAME + deployment.getScopeKernelName() + ",name=" + ejbName;
      try
      {
         objectName = new ObjectName(on);
      }
      catch (MalformedObjectNameException e)
      {
         throw new RuntimeException(e); // To change body of catch statement
         // use Options | File Templates.
      }
      initialContextProperties = ctxProperties;
      Context ctx = getInitialContext();
      try
      {
         enc = ThreadLocalENCFactory.create(ctx);
         encEnv = Util.createSubcontext(enc, "env");
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
      this.interceptorRepository = interceptorRepository;
      this.interceptorRepository.addBeanClass(clazz.getName());
   }

   public HashMap<String, String> getEjbRefDependencies()
   {
      return ejbRefDependencies;
   }

   public HashMap<String, String> getPuEncXmlEntries()
   {
      return puEncXmlEntries;
   }

   public HashMap<String, PersistenceContextRef> getPcEncXmlEntries()
   {
      return pcEncXmlEntries;
   }

   public String getJaccContextId()
   {
      return jaccContextId;
   }

   public void setJaccContextId(String jaccContextId)
   {
      this.jaccContextId = jaccContextId;
   }

   public Ejb3Deployment getDeployment()
   {
      return deployment;
   }

   public DependencyPolicy getDependencyPolicy()
   {
      return dependencyPolicy;
   }

   /**
    * EJBContainer has finished with all metadata initialization from XML files and such.
    * this is really a hook to do some processing after XML has been set up and before
    * and processing of dependencies and such.
    */
   public void instantiated()
   {

   }

   /**
    * introspects EJB container to find all dependencies
    * and initialize any extra metadata.
    *
    * This must be called before container is registered with any microcontainer
    *
    * @param dependencyPolicy
    */
   public void processMetadata(DependencyPolicy dependencyPolicy)
   {
      this.dependencyPolicy = dependencyPolicy;
      PersistenceContextHandler.loadDependencies(xml, this, clazz, true);
      PersistenceUnitHandler.loadDependencies(xml, this, clazz, true);
      if (xml != null)
      {
         for (String depends : xml.getDependencies())
         {
            dependencyPolicy.addDependency(depends);
         }
      }
      DependsHandler.loadDependencies(this, clazz);
      EJBHandler.loadDependencies(xml, this, clazz, true);
      initialiseInterceptors();
      for (InterceptorInfo interceptorInfo : applicableInterceptors)
      {
         PersistenceContextHandler.loadDependencies(interceptorInfo.getXml(), this, interceptorInfo.getClazz(), false);
         PersistenceUnitHandler.loadDependencies(interceptorInfo.getXml(), this, interceptorInfo.getClazz(), false);
         DependsHandler.loadDependencies(this, interceptorInfo.getClazz());
         EJBHandler.loadDependencies(interceptorInfo.getXml(), this, interceptorInfo.getClazz(), true);
      }
   }

   public EnterpriseBean getXml()
   {
      return xml;
   }

   public void setXml(EnterpriseBean xml)
   {
      this.xml = xml;
   }

   public AssemblyDescriptor getAssemblyDescriptor()
   {
      return assembly;
   }

   public void setAssemblyDescriptor(AssemblyDescriptor assembly)
   {
      this.assembly = assembly;
   }

   public InterceptorInfoRepository getInterceptorRepository()
   {
      return interceptorRepository;
   }

   public List<InterceptorInfo> getClassInterceptors()
   {
      initialiseInterceptors();
      return classInterceptors;
   }

   public HashSet<InterceptorInfo> getApplicableInterceptors()
   {
      initialiseInterceptors();
      return applicableInterceptors;
   }

   public HashMap<Class, InterceptorInjector> getInterceptorInjectors()
   {
      initialiseInterceptors();
      return interceptorInjectors;
   }

   public ClassLoader getClassloader()
   {
      return classloader;
   }

   public HashMap<String, String> getEncLinkRefEntries()
   {
      return encLinkRefEntries;
   }

   public void addEncLinkRefEntry(String name, String mappedName)
   {
      encLinkRefEntries.put(name, mappedName);
   }

   public void addEncInjector(AccessibleObject acc, Injector inj)
   {
      encInjections.put(acc, inj);
   }

   public InitialContext getInitialContext()
   {
      try
      {
         if (initialContextProperties == null)
            return new InitialContext();
         else
            return new InitialContext(initialContextProperties);
      }
      catch (NamingException e)
      {
         throw new RuntimeException(e);
      }
   }

   public HashMap<AccessibleObject, Injector> getEncInjections()
   {
      return encInjections;
   }

   public boolean hasEnvEntry(String name)
   {
      return envEntries.containsKey(name);
   }

   public void addEnvEntry(String name, String type, String value) throws ClassNotFoundException
   {
         envEntries.put(name, getEnvEntryValue(name, type, value));
   }

   public boolean hasEncEntry(String name)
   {
      return encLinkRefEntries.containsKey(name);
   }

   public Context getEnc()
   {
      return enc;
   }

   public Context getEncEnv()
   {
      return enc;
   }

   protected Object getEnvEntryValue(String name, String entryType, String value) throws ClassNotFoundException
   {
      Class type = this.classloader.loadClass(entryType);
      if (type == String.class)
      {
         return value;
      }
      else if (type == Integer.class)
      {
         return new Integer(value);
      }
      else if (type == Long.class)
      {
         return new Long(value);
      }
      else if (type == Double.class)
      {
         return new Double(value);
      }
      else if (type == Float.class)
      {
         return new Float(value);
      }
      else if (type == Byte.class)
      {
         return new Byte(value);
      }
      else if (type == Character.class)
      {
         String input = value;
         if (input == null || input.length() == 0)
         {
            return new Character((char) 0);
         }
         else
         {
            if (input.length() > 1)
               // TODO: Add deployment context
               log.warn("Warning character env-entry is too long: binding="
                        + name + " value=" + input);
            return new Character(input.charAt(0));
         }
      }
      else if (type == Short.class)
      {
         return new Short(value);
      }
      else if (type == Boolean.class)
      {
         return new Boolean(value);
      }
      else
      {
         return value;
      }
   }

   public Hashtable getInitialContextProperties()
   {
      return initialContextProperties;
   }

   public ObjectName getObjectName()
   {
      return objectName;
   }

   public String getEjbName()
   {
      return ejbName;
   }

   public String getBeanClassName()
   {
      return beanClassName;
   }

   public Class getBeanClass()
   {
      return clazz;
   }

   public Pool getPool()
   {
      return pool;
   }

   public Object construct()
   {
      Interceptor[] cInterceptors = constructorInterceptors[defaultConstructorIndex];
      if (cInterceptors == null)
      {
         try
         {
            return constructors[defaultConstructorIndex].newInstance();
         }
         catch (InstantiationException e)
         {
            throw new RuntimeException(e);
         }
         catch (IllegalAccessException e)
         {
            throw new RuntimeException(e);
         }
         catch (InvocationTargetException e)
         {
            throw new RuntimeException(e);
         }
      }
      ConstructorInvocation invocation = new ConstructorInvocation(
              cInterceptors);

      invocation.setAdvisor(this);
      invocation.setConstructor(constructors[defaultConstructorIndex]);
      try
      {
         return invocation.invokeNext();
      }
      catch (Throwable throwable)
      {
         throw new RuntimeException(throwable);
      }

   }

   public void create() throws Exception
   {
   }

   // Everything must be done in start to make sure all dependencies have been satisfied
   public void start() throws Exception
   {
      initializeClassContainer();
      for (int i = 0; i < constructors.length; i++)
      {
         if (constructors[i].getParameterTypes().length == 0)
         {
            defaultConstructorIndex = i;
            break;
         }
      }
      createEnvEntries();
      initializePool();
      resolveInterceptorInjectors();

      // add extendedPC injectors to be first in list
      ArrayList injectors2 = new ArrayList();
      injectors2.addAll(extendedPCs.values());
      if (injectors != null) injectors2.addAll(Arrays.asList(injectors));
      injectors = (Injector[])injectors2.toArray(new Injector[injectors2.size()]);
      if (pool != null) pool.setInjectors(injectors);

      createCallbackHandler();

      for (String encName : encLinkRefEntries.keySet())
      {
         String mappedName = encLinkRefEntries.get(encName);
         try
         {
            Util.bind(enc, encName, new LinkRef(mappedName));
            log.debug("binding enc: " + encName + " to " + mappedName);
         } catch (NamingException e)
         {
            NamingException namingException = new NamingException("Could not bind EJB container with ejb name " + ejbName + " into JNDI under jndiName: " + enc.getNameInNamespace() + "/" + encName);
            namingException.setRootCause(e);
            throw namingException;
         }
      }
      JaccHelper.configureContainer(jaccContextId, this);
      log.info("STARTED EJB: " + clazz.getName() + " ejbName: " + ejbName);
   }

   public void stop() throws Exception
   {
   }

   public void destroy() throws Exception
   {
   }

   protected void createEnvEntries() throws NamingException
   {
      Iterator names = envEntries.keySet().iterator();
      while (names.hasNext())
      {
         String name = (String) names.next();
         Object value = envEntries.get(name);
              
         try 
         {
            Util.bind(encEnv,
                      name,
                      value);
         } catch (NamingException e)
         {
            NamingException namingException = new NamingException("Could not bind env entry for ejb name " + ejbName + " into JNDI under jndiName: " + encEnv.getNameInNamespace() + "/" + name);
            namingException.setRootCause(e);
            throw namingException;
         }
      }
   }

   public void initializePool() throws Exception
   {
      PoolClass poolClass = (PoolClass) resolveAnnotation(PoolClass.class);
      Class poolClazz = poolClass.value();
      int maxSize = poolClass.maxSize();
      long timeout = poolClass.timeout();
      pool = (Pool) poolClazz.newInstance();
      pool.initialize(this, beanContextClass, clazz, maxSize, timeout);

      resolveInjectors();
      pool.setInjectors(injectors);
   }

   public void invokePostConstruct(BeanContext beanContext)
   {
      callbackHandler.postConstruct(beanContext);
   }

   public void invokePreDestroy(BeanContext beanContext)
   {
      callbackHandler.preDestroy(beanContext);
   }

   public void invokePostActivate(BeanContext beanContext)
   {
      throw new RuntimeException("PostActivate not implemented for container");
   }

   public void invokePrePassivate(BeanContext beanContext)
   {
      throw new RuntimeException("PostActivate not implemented for container");
   }

   public void invokeInit(Object bean)
   {

   }

   public void invokeInit(Object bean, Class[] initParameterTypes,
                          Object[] initParameterValues)
   {

   }

   public static final String MANAGED_ENTITY_MANAGER_FACTORY = "ManagedEntityManagerFactory";

   public static final String ENTITY_MANAGER_FACTORY = "EntityManagerFactory";

   public HashMap getExtendedPCs()
   {
      return extendedPCs;
   }

   protected void resolveInjectors() throws Exception
   {
      ThreadLocalENCFactory.push(enc);
      try
      {
         Thread.currentThread().setContextClassLoader(classloader);
         try
         {
            Util.bind(enc, "UserTransaction", new UserTransactionImpl());
         } catch (NamingException e)
         {
            NamingException namingException = new NamingException("Could not bind user transaction for ejb name " + ejbName + " into JNDI under jndiName: " + enc.getNameInNamespace() + "/" + "UserTransaction");
            namingException.setRootCause(e);
            throw namingException;
         }

         List list = new ArrayList();
         PersistenceContextHandler.loadInjectors(this);
         PersistenceUnitHandler.loadInjectors(this);
         list.addAll(JndiInjectHandler.loadInjectors(this));
         ResourceHandler.loadInjectors(this);
         WebServiceHandler.loadInjectors(this);
         EJBHandler.loadInjectors(this);
         list.addAll(DependsHandler.loadInjectors(this));
         list.addAll(encInjections.values());
         injectors = (Injector[]) list.toArray(new Injector[list.size()]);
      }
      finally
      {
         ThreadLocalENCFactory.pop();
      }
   }

   protected void resolveInterceptorInjectors() throws Exception
   {
      ThreadLocalENCFactory.push(enc);
      try
      {
         Thread.currentThread().setContextClassLoader(classloader);
         if (applicableInterceptors != null && applicableInterceptors.size() > 0)
         {
            interceptorInjectors = new HashMap<Class, InterceptorInjector>();
            for (InterceptorInfo interceptorInfo : applicableInterceptors)
            {
               InterceptorInjector injector = new InterceptorInjector(this, interceptorInfo);
               interceptorInjectors.put(interceptorInfo.getClazz(), injector);
            }
         }
      }
      finally
      {
         ThreadLocalENCFactory.pop();
      }
   }

   protected void createCallbackHandler()
   {
      try
      {
         callbackHandler = new LifecycleInterceptorHandler(this,
                                                           getHandledCallbacks());
      }
      catch (Exception e)
      {
         throw new RuntimeException("Error creating callback handler for bean "
                                    + beanClassName, e);
      }
   }

   protected Class[] getHandledCallbacks()
   {
      return new Class[]
              {PostConstruct.class, PreDestroy.class, Timeout.class};
   }

   private void initialiseInterceptors()
   {
      if (applicableInterceptors == null)
      {
         log.debug("Initialising interceptors for " + getEjbName() + "...");
         HashSet<InterceptorInfo> defaultInterceptors = interceptorRepository.getDefaultInterceptors();
         log.debug("Default interceptors: " + defaultInterceptors);

         classInterceptors = interceptorRepository.getClassInterceptors(this);
         log.debug("Class interceptors: " + classInterceptors);

         applicableInterceptors = new LinkedHashSet<InterceptorInfo>();
         if (defaultInterceptors != null) applicableInterceptors.addAll(defaultInterceptors);
         if (classInterceptors != null) applicableInterceptors.addAll(classInterceptors);

         Method[] methods = clazz.getMethods();
         for (int i = 0; i < methods.length; i++)
         {
            List methodIcptrs = interceptorRepository.getMethodInterceptors(this, methods[i]);
            if (methodIcptrs != null && methodIcptrs.size() > 0)
            {
               log.debug("Method interceptors for  " + methods[i] + ": " + methodIcptrs);
               applicableInterceptors.addAll(methodIcptrs);
            }
         }
         log.debug("All applicable interceptor classes: " + applicableInterceptors);
      }
   }
   
   public Object getBusinessObject(BeanContext beanContext, Class businessObject) throws IllegalStateException
   {
      throw new IllegalStateException("Not implemented");
   }
   
   public Object getInvokedBusinessInterface(BeanContext beanContext) throws IllegalStateException
   {
      throw new IllegalStateException("Not implemented");
   }

   protected Object getInvokedInterface(Method method)
   {
      Remote remoteAnnotation = (Remote)resolveAnnotation(Remote.class);
      if (remoteAnnotation != null)
      {
         Class[] remotes = remoteAnnotation.value();
         for (int i = 0 ; i < remotes.length ; ++i)
         {
            try
            {
               remotes[i].getMethod(method.getName(), method.getParameterTypes());
               return remotes[i];
            }
            catch (NoSuchMethodException e)
            {}
         }
      }
      
      Local localAnnotation = (Local)resolveAnnotation(Local.class);
      if (localAnnotation != null)
      {
         Class[] locals = localAnnotation.value();
         for (int i = 0 ; i < locals.length ; ++i)
         {
            Method[] interfaceMethods = locals[i].getMethods();
            for ( int j = 0 ; j < interfaceMethods.length ; ++j)
            {
               if (interfaceMethods[j].equals(method))
                  return locals[i];
            }
         }
      }
      
      return null;
   }

   // todo these method overrides for aop are for performance reasons
   private Class loadPublicAnnotation(String annotation)
   {
      try
      {
         Class ann = classloader.loadClass(annotation);
         if (!ann.isAnnotation()) return null;
         Retention retention = (Retention)ann.getAnnotation(Retention.class);
         if (retention != null && retention.value() == RetentionPolicy.RUNTIME) return ann;

      }
      catch (ClassNotFoundException ignored)
      {
      }
      return null;
   }

   @Override
   public boolean hasAnnotation(Class tgt, String annotation)
   {
      if (annotations.hasClassAnnotation(annotation)) return true;
      if (tgt == null) return false;
      try
      {
         Class ann = loadPublicAnnotation(annotation);
         // it is metadata or CLASS annotation
         if (ann == null) return AnnotationElement.isAnyAnnotationPresent(tgt, annotation);
         return tgt.isAnnotationPresent(ann);
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);  //To change body of catch statement use Options | File Templates.
      }
   }


   @Override
   public boolean hasAnnotation(Method m, String annotation)
   {

      if (annotations.hasAnnotation(m, annotation)) return true;
      try
      {
         Class ann = loadPublicAnnotation(annotation);
         // it is metadata or CLASS annotation
         if (ann == null) return AnnotationElement.isAnyAnnotationPresent(m, annotation);
         return m.isAnnotationPresent(ann);
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);  //To change body of catch statement use Options | File Templates.
      }
   }

   @Override
   public boolean hasAnnotation(Field m, String annotation)
   {
      if (annotations.hasAnnotation(m, annotation)) return true;
      try
      {
         Class ann = loadPublicAnnotation(annotation);
         // it is metadata or CLASS annotation
         if (ann == null) return AnnotationElement.isAnyAnnotationPresent(m, annotation);
         return m.isAnnotationPresent(ann);
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);  //To change body of catch statement use Options | File Templates.
      }
   }

   @Override
   public boolean hasAnnotation(Constructor m, String annotation)
   {
      if (annotations.hasAnnotation(m, annotation)) return true;
      try
      {
         Class ann = loadPublicAnnotation(annotation);
         // it is metadata or CLASS annotation
         if (ann == null) return AnnotationElement.isAnyAnnotationPresent(m, annotation);
         return m.isAnnotationPresent(ann);
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);  //To change body of catch statement use Options | File Templates.
      }
   }
}
