/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.xml.binding;

import org.jboss.logging.Logger;
import org.jboss.xml.binding.parser.JBossXBParser;
import org.jboss.xml.binding.parser.xni.XniJBossXBParser;
import org.jboss.xml.binding.parser.sax.SaxJBossXBParser;
import org.jboss.util.xml.JBossEntityResolver;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;

import java.io.Reader;
import java.io.InputStream;

/**
 * Unmarshaller implementation.
 * WARNING: this implementation is not thread-safe.
 *
 * @version <tt>$Revision$</tt>
 * @author <a href="mailto:alex@jboss.org">Alexey Loubyansky</a>
 */
public class Unmarshaller
{
   private static final Logger log = Logger.getLogger(Unmarshaller.class);

   public static final String VALIDATION = "http://xml.org/sax/features/validation";
   public static final String NAMESPACES = "http://xml.org/sax/features/namespaces";
   public static final String NAMESPACE_PREFIXES = "http://xml.org/sax/features/namespace-prefixes";

   // set some xerces specific features that allow transparent DTD and Schema validation
   public static final String DYNAMIC_VALIDATION = "http://apache.org/xml/features/validation/dynamic";
   public static final String SCHEMA_VALIDATION = "http://apache.org/xml/features/validation/schema";
   public static final String SCHEMA_FULL_CHECKING = "http://apache.org/xml/features/validation/schema-full-checking";

   private ObjectModelBuilder builder = new ObjectModelBuilder();
   private final JBossXBParser parser;

   // Constructor

   /**
    * The constructor for DTD and XSD client awareness. Invokes this(false).
    */
   public Unmarshaller()
      throws JBossXBException
   {
      this(false);
   }
   /**
    * Create a parser with a choice between the XniJBossXBParser and
    * SaxJBossXBParser.
    * 
    * @param useXniParser - true to use the XniJBossXBParser, false to use
    *    the SaxJBossXBParser
    * @throws JBossXBException
    */ 
   public Unmarshaller(boolean useXniParser)
      throws JBossXBException
   {
      if( useXniParser == true )
         parser = new XniJBossXBParser();
      else
         parser = new SaxJBossXBParser();

      parser.setFeature(VALIDATION, true);
      parser.setFeature(SCHEMA_VALIDATION, true);
      parser.setFeature(SCHEMA_FULL_CHECKING, true);
      parser.setFeature(DYNAMIC_VALIDATION, true);
      parser.setFeature(NAMESPACES, true);

      parser.setEntityResolver(new JBossEntityResolver());
   }

   public void setValidation(boolean validation)
      throws JBossXBException
   {
      parser.setFeature(VALIDATION, validation);
   }

   public void setNamespaceAware(boolean namespaces)
      throws JBossXBException
   {
      parser.setFeature(NAMESPACES, namespaces);
   }

   public void setEntityResolver(EntityResolver entityResolver) throws JBossXBException
   {
      parser.setEntityResolver(entityResolver);
   }

   public void setErrorHandler(ErrorHandler errorHandler)
   {
      // todo reader.setErrorHandler(errorHandler);
   }

   public void mapFactoryToNamespace(ObjectModelFactory factory, String namespaceUri)
   {
      builder.mapFactoryToNamespace(getGenericObjectModelFactory(factory), namespaceUri);
   }

   public Object unmarshal(Reader reader, ObjectModelFactory factory, Object root) throws JBossXBException
   {
      builder.init(getGenericObjectModelFactory(factory), root);
      parser.parse(reader, builder);
      return builder.getRoot();
   }

   public Object unmarshal(InputStream is, ObjectModelFactory factory, Object root) throws JBossXBException
   {
      builder.init(getGenericObjectModelFactory(factory), root);
      parser.parse(is, builder);
      return builder.getRoot();
   }

   public Object unmarshal(String systemId, ObjectModelFactory factory, Object root)
      throws JBossXBException
   {
      builder.init(getGenericObjectModelFactory(factory), root);
      parser.parse(systemId, builder);
      return builder.getRoot();
   }
   public Object unmarshal(InputSource is, ObjectModelFactory factory, Object root) throws JBossXBException
   {
      Object result;
      if(is.getCharacterStream() != null)
      {
         result = unmarshal(is.getCharacterStream(), factory, root);
      }
      else if(is.getByteStream() != null)
      {
         result = unmarshal(is.getByteStream(), factory, root);
      }
      else
      {
         result = unmarshal(is.getSystemId(), factory, root);
      }
      return result;
   }

   private static final GenericObjectModelFactory getGenericObjectModelFactory(ObjectModelFactory factory)
   {
      if(!(factory instanceof GenericObjectModelFactory))
      {
         factory = new DelegatingObjectModelFactory(factory);
      }
      return factory instanceof GenericObjectModelFactory ? (GenericObjectModelFactory)factory : new DelegatingObjectModelFactory(factory);
   }
}
