/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

// $Id:DOMUtils.java,v 1.0, 2005-06-24 19:01:50Z, Robert Worsnop$
package org.jboss.webservice.util;

// $Id:DOMUtils.java,v 1.0, 2005-06-24 19:01:50Z, Robert Worsnop$

import org.jboss.logging.Logger;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.xml.namespace.QName;

/**
 * A utility class for common w3c DOM operations.
 *
 * @author Thomas.Diesler@jboss.org
 * @since 15-May-2004
 */
public final class DOMUtils
{
   // provide logging
   private static final Logger log = Logger.getLogger(DOMUtils.class);

   // hide the constructor
   private DOMUtils()
   {
   }

   /**
    * Get the value from the given attribute
    *
    * @return null if the attribute value is empty or the attribute is not present
    */
   public static String getAttributeValue(Element el, String attrName)
   {
      String attr = el.getAttribute(attrName);

      if ("".equals(attr))
         attr = null;

      return attr;
   }

   /**
    * Get the qname value from the given attribute
    */
   public static QName getAttributeValueAsQName(Element el, String attrName)
   {
      QName qname = null;

      String attr = getAttributeValue(el, attrName);
      if (attr != null)
      {
         String prefix = "";
         String localPart = attr;
         String namespaceURI = "";

         int colonIndex = attr.indexOf(":");
         if (colonIndex > 0)
         {
            prefix = attr.substring(0, colonIndex);
            localPart = attr.substring(colonIndex + 1);

            Element nsElement = el;
            while (namespaceURI.equals("") && nsElement != null)
            {
               namespaceURI = nsElement.getAttribute("xmlns:" + prefix);
               if (namespaceURI.equals(""))
                  nsElement = getParentElement(nsElement);
            }

            if (namespaceURI.equals(""))
               throw new IllegalArgumentException("Cannot find namespace uri for: " + attr);
         }

         qname = new QName(namespaceURI, localPart, prefix);
      }

      return qname;
   }

   /**
    * Get the boolean value from the given attribute
    */
   public static boolean getAttributeValueAsBoolean(Element el, String attrName)
   {
      String attrVal = getAttributeValue(el, attrName);
      boolean ret = "true".equalsIgnoreCase(attrVal) || "1".equalsIgnoreCase(attrVal);
      return ret;
   }

   /**
    * Gets the first child element
    */
   public static Element getFirstChildElement(Element el)
   {
      return getFirstChildElement(el, null);
   }

   /**
    * Gets the first child element for a givenname
    */
   public static Element getFirstChildElement(Element el, String tagName)
   {
      Element childElement = null;
      NodeList nlist = el.getChildNodes();
      for (int i = 0; childElement == null && i < nlist.getLength(); i++)
      {
         Node node = nlist.item(i);
         if (node.getNodeType() == Node.ELEMENT_NODE)
         {
            if (tagName == null || tagName.equals(node.getLocalName()))
               childElement = (Element)node;
         }
      }
      return childElement;
   }

   /**
    * Gets parent element or null if there is none
    */
   public static Element getParentElement(Element el)
   {
      Node parent = el.getParentNode();
      return (parent instanceof Element ? (Element)parent : null);
   }
}