/**
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.webservice.handler;

// $Id:HandlerWrapper.java,v 1.0, 2005-06-24 19:04:59Z, Robert Worsnop$

import org.jboss.logging.Logger;

import javax.xml.namespace.QName;
import javax.xml.rpc.JAXRPCException;
import javax.xml.rpc.handler.Handler;
import javax.xml.rpc.handler.HandlerInfo;
import javax.xml.rpc.handler.MessageContext;
import javax.xml.rpc.soap.SOAPFaultException;

/**
 * A wrapper arround a {@link javax.xml.rpc.handler.Handler} that takes care of its lifecycle.
 *
 * @author thomas.diesler@jboss.org
 */
public class HandlerWrapper implements Handler
{
   private static Logger log = Logger.getLogger(HandlerWrapper.class);

   public final static int DOES_NOT_EXIST = 0;
   public final static int METHOD_READY = 1;

   // The states as string
   private static String[] stateNames = new String[]{"DOES_NOT_EXIST", "METHOD_READY"};

   // The handler to delegate to
   private Handler delegate;
   // The handler state
   private int state;

   /**
    * Delegate to the given handler
    */
   public HandlerWrapper(Handler handler)
   {
      delegate = handler;
      state = DOES_NOT_EXIST; // this is somewhat a lie ;-)
   }

   /**
    * Get the current state
    */
   public int getState()
   {
      return state;
   }

   /**
    * Get the current state as string
    */
   public String getStateAsString()
   {
      return stateNames[state];
   }

   /**
    * Gets the header blocks processed by this Handler instance.
    */
   public QName[] getHeaders()
   {
      return delegate.getHeaders();
   }

   /**
    * The init method enables the Handler instance to initialize itself.
    */
   public void init(HandlerInfo config) throws JAXRPCException
   {
      log.debug("init: " + delegate);
      delegate.init(config);
      state = METHOD_READY;
   }

   /**
    * The destroy method indicates the end of lifecycle for a Handler instance.
    */
   public void destroy() throws JAXRPCException
   {
      log.debug("destroy: " + delegate);
      state = DOES_NOT_EXIST;
      delegate.destroy();
   }

   /**
    * The handleRequest method processes the request message.
    */
   public boolean handleRequest(MessageContext msgContext) throws JAXRPCException, SOAPFaultException
   {
      if (state == DOES_NOT_EXIST)
      {
         log.warn("Handler is in state DOES_NOT_EXIST, skipping Handler.handleRequest for: " + delegate);
         return true;
      }

      try
      {
         return delegate.handleRequest(msgContext);
      }
      catch (RuntimeException e)
      {
         return handleRuntimeException(e);
      }
   }

   /**
    * The handleResponse method processes the response SOAP message.
    */
   public boolean handleResponse(MessageContext msgContext)
   {
      if (state == DOES_NOT_EXIST)
      {
         log.warn("Handler is in state DOES_NOT_EXIST, skipping Handler.handleResponse for: " + delegate);
         return true;
      }

      try
      {
         return delegate.handleResponse(msgContext);
      }
      catch (RuntimeException e)
      {
         return handleRuntimeException(e);
      }
   }

   /**
    * The handleFault method processes the SOAP faults based on the SOAP message processing model.
    */
   public boolean handleFault(MessageContext msgContext)
   {
      if (state == DOES_NOT_EXIST)
      {
         log.warn("Handler is in state DOES_NOT_EXIST, skipping Handler.handleFault for: " + delegate);
         return true;
      }

      try
      {
         return delegate.handleFault(msgContext);
      }
      catch (RuntimeException e)
      {
         return handleRuntimeException(e);
      }
   }

   /**
    * As defined by JAX-RPC, a RuntimeException(other than SOAPFaultException) thrown from any method of
    * the Handler results in the destroymethod being invoked and transition to the Does Not Exist state.
    */
   private boolean handleRuntimeException(RuntimeException e)
   {
      if ((e instanceof SOAPFaultException) == false)
      {
         log.warn("RuntimeException in handler method, transition to DOES_NOT_EXIST");
         destroy();
      }

      throw e;
   }

   /**
    * Returns a hash code value for the object.
    */
   public int hashCode()
   {
      return delegate.hashCode();
   }

   /**
    * Returns a string representation of the object.
    */
   public String toString()
   {
      return "[state=" + getStateAsString() + ",handler=" + delegate + "]";
   }
}
