/*
 * JBoss, the OpenSource WebOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.web.tomcat.tc5.jca;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.resource.ResourceException;
import javax.servlet.ServletException;
import javax.transaction.Status;
import javax.transaction.SystemException;
import javax.transaction.TransactionManager;

import org.apache.catalina.Lifecycle;
import org.apache.catalina.LifecycleException;
import org.apache.catalina.LifecycleListener;
import org.apache.catalina.Wrapper;
import org.apache.catalina.connector.Request;
import org.apache.catalina.connector.Response;
import org.apache.catalina.valves.ValveBase;
import org.apache.catalina.util.LifecycleSupport;
import org.jboss.logging.Logger;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.resource.connectionmanager.CachedConnectionManager;

/**
 * This valve checks for unclosed connections on a servlet request
 *
 * @author <a href="mailto:adrian@jboss.com">Adrian Brock</a>
 * @version $Revision:1$
 */
public class CachedConnectionValve extends ValveBase implements Lifecycle
{
   /**
    * The log
    */
   private static final Logger log = Logger.getLogger(CachedConnectionValve.class);

   /**
    * The info string for this Valve
    */
   private static final String info = "CachedConnectionValve/1.0";

   /**
    * Valve-lifecycle helper object
    */
   protected LifecycleSupport support = new LifecycleSupport(this);

   /**
    * The object name of the cached connection manager
    */
   protected String ccmName;

   /**
    * The cached connection manager
    */
   protected CachedConnectionManager ccm;

   /**
    * The object name of the transaction manager service
    */
   protected String tmName;

   /**
    * The transaction manager
    */
   protected TransactionManager tm;

   /**
    * The unshareable resources
    */
   protected Set unsharableResources = new HashSet();

   /**
    * Create a new valve
    *
    * @param ccm the cached connection manager for the valve
    */
   public CachedConnectionValve()
   {
      super();
   }

   /**
    * Get information about this Valve.
    */
   public String getInfo()
   {
      return info;
   }

   /**
    * Get the cached connection manager object name
    */
   public String getCachedConnectionManagerObjectName()
   {
      return ccmName;
   }

   /**
    * Set the cached connection manager object name
    */
   public void setCachedConnectionManagerObjectName(String ccmName)
   {
      this.ccmName = ccmName;
   }

   /**
    * Get the transaction manager object name
    */
   public String getTransactionManagerObjectName()
   {
      return tmName;
   }

   /**
    * Set the transaction manager object name
    */
   public void setTransactionManagerObjectName(String tmName)
   {
      this.tmName = tmName;
   }

   public void invoke(Request request, Response response) throws IOException, ServletException
   {
      try
      {
         ccm.pushMetaAwareObject(this, unsharableResources);
         try
         {
            getNext().invoke(request, response);
         }
         finally
         {
            ccm.popMetaAwareObject(unsharableResources);
            checkTransactionComplete(request);
         }
      }
      catch (ResourceException e)
      {
         throw new ServletException("Error invoking cached connection manager", e);
      }
   }

   // Lifecycle-interface
   public void addLifecycleListener(LifecycleListener listener)
   {
      support.addLifecycleListener(listener);
   }

   public void removeLifecycleListener(LifecycleListener listener)
   {
      support.removeLifecycleListener(listener);
   }

   public LifecycleListener[] findLifecycleListeners()
   {
      return support.findLifecycleListeners();
   }

   public void start() throws LifecycleException
   {
      try
      {
         MBeanServer server = MBeanServerLocator.locateJBoss();
         ccm = (CachedConnectionManager) server.getAttribute(new ObjectName(ccmName), "Instance");
         tm = (TransactionManager) server.getAttribute(new ObjectName(tmName), "TransactionManager");
      }
      catch (Exception e)
      {
         throw new LifecycleException(e);
      }
      
      // TODO unshareable resources
      support.fireLifecycleEvent(START_EVENT, this);
   }

   public void stop() throws LifecycleException
   {
      support.fireLifecycleEvent(STOP_EVENT, this);
      unsharableResources.clear();
   }

   protected void checkTransactionComplete(Request request)
   {
      int status = Status.STATUS_NO_TRANSACTION;

      try
      {
         status = tm.getStatus();
      }
      catch (SystemException ex)
      {
         log.error("Failed to get status", ex);
      }

      switch (status)
      {
         case Status.STATUS_ACTIVE:
         case Status.STATUS_COMMITTING:
         case Status.STATUS_MARKED_ROLLBACK:
         case Status.STATUS_PREPARING:
         case Status.STATUS_ROLLING_BACK:
            try
            {
               tm.rollback();
            }
            catch (Exception ex)
            {
               log.error("Failed to rollback", ex);
            }
            // fall through...
         case Status.STATUS_PREPARED:
            String servletName = "<Unknown>";
            try
            {
               Wrapper servlet = request.getWrapper();
               if (servlet != null)
               {
                  servletName = servlet.getName();
                  if (servlet.getJspFile() != null)
                     servletName = servlet.getJspFile();
               }
            }
            catch (Throwable ignored)
            {
            }

            String msg = "Application error: " + servletName + " did not complete its transaction";
            log.error(msg);
      }
   }
}
