/*
 * JBoss, the OpenSource EJB server
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.test.security.test;

import java.util.Map;
import java.util.Iterator;
import java.util.Set;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.security.acl.Group;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.AppConfigurationEntry.LoginModuleControlFlag;
import javax.security.auth.Subject;

import org.apache.log4j.Logger;

import org.jboss.logging.XLevel;
import org.jboss.security.auth.login.XMLLoginConfigImpl;
import org.jboss.security.auth.spi.Users;
import org.jboss.security.auth.login.LoginConfigObjectModelFactory;
import org.jboss.security.auth.login.PolicyConfig;
import org.jboss.security.auth.login.AuthenticationInfo;
import org.jboss.security.auth.spi.UsersObjectModelFactory;
import org.jboss.security.auth.callback.UsernamePasswordHandler;
import org.jboss.security.SimplePrincipal;
import org.jboss.xml.binding.Unmarshaller;

/**
 * Tests of the LoginModule classes using the XMLLoginConfigImpl implementation
 * of the JAAS login module configuration.
 * @author Scott.Stark@jboss.org
 * @version $Revision:1$
 */
public class XMLLoginModulesUnitTestCase extends LoginModulesUnitTestCase
{

   public XMLLoginModulesUnitTestCase(String name)
   {
      super(name);
   }

   protected void setUp() throws Exception
   {
      // Setup the replacement properties
      System.setProperty("users.properties", "/security/config/users.properites");
      System.setProperty("roles.properties", "/security/config/roles.properites");

      // Install the custom JAAS configuration
      XMLLoginConfigImpl config = new XMLLoginConfigImpl();
      config.setConfigResource("security/login-config.xml");
      config.loadConfig();
      Configuration.setConfiguration(config);

      // Turn on trace level logging
      Logger root = Logger.getRootLogger();
      root.setLevel(XLevel.TRACE);
   }

   public void testPropertyReplacement() throws Exception
   {
      Configuration config = Configuration.getConfiguration();
      AppConfigurationEntry[] entries = config.getAppConfigurationEntry("testPropertyReplacement");
      assertTrue("entries.length == 1", entries.length == 1);
      AppConfigurationEntry entry = entries[0];
      LoginModuleControlFlag flag = entry.getControlFlag();
      assertTrue("flag == required", flag == LoginModuleControlFlag.REQUIRED);
      Map options = entry.getOptions();
      String users = (String) options.get("usersProperties");
      assertTrue("usersProperties(" + users + ") = /security/config/users.properites",
         users.equals("/security/config/users.properites"));
      String roles = (String) options.get("rolesProperties");
      assertTrue("rolesProperties(" + roles + ") = /security/config/roles.properites",
         roles.equals("/security/config/roles.properites"));
   }

   /**
    * @throws Exception
    */
   public void testXmlLoginModuleParsing() throws Exception
   {
      LoginConfigObjectModelFactory lcomf = new LoginConfigObjectModelFactory();
      UsersObjectModelFactory uomf = new UsersObjectModelFactory();

      InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream("security/login-config2.xml");
      InputStreamReader xmlReader = new InputStreamReader(is);
      Unmarshaller unmarshaller = new Unmarshaller();
      unmarshaller.mapFactoryToNamespace(uomf, "http://www.jboss.org/j2ee/schemas/XMLLoginModule");
      PolicyConfig config = (PolicyConfig) unmarshaller.unmarshal(xmlReader, lcomf, null);
      AuthenticationInfo info = config.get("testXMLLoginModule");
      assertTrue("test-xml-config != null", info != null);
      AppConfigurationEntry[] entries = info.getAppConfigurationEntry();
      assertTrue("entries.length == 1", entries.length == 1);
      AppConfigurationEntry ace = entries[0];
      assertTrue("org.jboss.security.auth.spi.XMLLoginModule",
         ace.getLoginModuleName().equals("org.jboss.security.auth.spi.XMLLoginModule"));
      Map options = ace.getOptions();
      assertTrue("Options.size == 2", options.size() == 2);
      String guest = (String) options.get("unauthenticatedIdentity");
      assertTrue("guest", guest.equals("guest"));
      Users users = (Users) options.get("userInfo");
      Users.User user = users.getUser("jdukeman");
      String name = user.getName();
      assertTrue("name == jdukeman", name.equals("jdukeman"));
      String passwrd = user.getPassword();
      assertTrue("passwrd == anotherduke", passwrd.equals("anotherduke"));
      String[] roleNames = user.getRoleNames();
      assertTrue("roles in (Role2, Role3)",
         roleNames[0].equals("Role2") && roleNames[1].equals("Role3"));
   }

   public void testXMLLoginModule() throws Exception
   {
      // Install the custom JAAS configuration
      XMLLoginConfigImpl config = new XMLLoginConfigImpl();
      config.setConfigResource("security/login-config2.xml");
      config.loadConfig();
      Configuration.setConfiguration(config);

      getLog().info("testXMLLoginModule");
      UsernamePasswordHandler handler = new UsernamePasswordHandler("scott", "echoman".toCharArray());
      LoginContext lc = new LoginContext("testXMLLoginModule", handler);
      lc.login();
      Subject subject = lc.getSubject();
      Set groups = subject.getPrincipals(Group.class);
      assertTrue("Principals contains scott", subject.getPrincipals().contains(new SimplePrincipal("scott")));
      assertTrue("Principals contains Roles", groups.contains(new SimplePrincipal("Roles")));
      assertTrue("Principals contains CallerPrincipal", groups.contains(new SimplePrincipal("CallerPrincipal")));
      Group roles = (Group) groups.iterator().next();
      Iterator groupsIter = groups.iterator();
      while (groupsIter.hasNext())
      {
         roles = (Group) groupsIter.next();
         if (roles.getName().equals("Roles"))
         {
            assertTrue("Echo is a role", roles.isMember(new SimplePrincipal("Echo")));
            assertTrue("Java is NOT a role", roles.isMember(new SimplePrincipal("Java")) == false);
            assertTrue("Coder is NOT a role", roles.isMember(new SimplePrincipal("Coder")) == false);
         }
         else if (roles.getName().equals("CallerPrincipal"))
         {
            getLog().info("CallerPrincipal is " + roles.members().nextElement());
            boolean isMember = roles.isMember(new SimplePrincipal("callerScott"));
            assertTrue("CallerPrincipal is callerScott", isMember);
         }
      }
      lc.logout();

      handler = new UsernamePasswordHandler("stark", "javaman".toCharArray());
      lc = new LoginContext("testXMLLoginModule", handler);
      lc.login();
      subject = lc.getSubject();
      groups = subject.getPrincipals(Group.class);
      assertTrue("Principals contains stark", subject.getPrincipals().contains(new SimplePrincipal("stark")));
      assertTrue("Principals contains Roles", groups.contains(new SimplePrincipal("Roles")));
      assertTrue("Principals contains CallerPrincipal", groups.contains(new SimplePrincipal("CallerPrincipal")));
      groupsIter = groups.iterator();
      while (groupsIter.hasNext())
      {
         roles = (Group) groupsIter.next();
         if (roles.getName().equals("Roles"))
         {
            assertTrue("Echo is NOT a role", roles.isMember(new SimplePrincipal("Echo")) == false);
            assertTrue("Java is a role", roles.isMember(new SimplePrincipal("Java")));
            assertTrue("Coder is a role", roles.isMember(new SimplePrincipal("Coder")));
         }
         else if (roles.getName().equals("CallerPrincipal"))
         {
            getLog().info("CallerPrincipal is " + roles.members().nextElement());
            boolean isMember = roles.isMember(new SimplePrincipal("callerStark"));
            assertTrue("CallerPrincipal is callerStark", isMember);
         }
      }
      lc.logout();

      // Test the usernames with common prefix
      getLog().info("Testing similar usernames");
      handler = new UsernamePasswordHandler("jdukeman", "anotherduke".toCharArray());
      lc = new LoginContext("testXMLLoginModule", handler);
      lc.login();
      subject = lc.getSubject();
      groups = subject.getPrincipals(Group.class);
      assertTrue("Principals contains jdukeman", subject.getPrincipals().contains(new SimplePrincipal("jdukeman")));
      assertTrue("Principals contains Roles", groups.contains(new SimplePrincipal("Roles")));
      assertTrue("Principals contains CallerPrincipal", groups.contains(new SimplePrincipal("CallerPrincipal")));
      groupsIter = groups.iterator();
      while (groupsIter.hasNext())
      {
         roles = (Group) groupsIter.next();
         if (roles.getName().equals("Roles"))
         {
            assertTrue("Role1 is NOT a role", roles.isMember(new SimplePrincipal("Role1")) == false);
            assertTrue("Role2 is a role", roles.isMember(new SimplePrincipal("Role2")));
            assertTrue("Role3 is a role", roles.isMember(new SimplePrincipal("Role3")));
         }
         else if (roles.getName().equals("CallerPrincipal"))
         {
            getLog().info("CallerPrincipal is " + roles.members().nextElement());
            boolean isMember = roles.isMember(new SimplePrincipal("callerJdukeman"));
            assertTrue("CallerPrincipal is callerJdukeman", isMember);
         }
      }
      lc.logout();
   }
}
