/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.test.media.entity.test;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.StringWriter;

import javax.emb.ContentAccessException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.rmi.PortableRemoteObject;

import junit.framework.Test;
import net.sourceforge.junitejb.EJBTestCase;

import org.jboss.logging.Logger;
import org.jboss.test.JBossTestCase;
import org.jboss.test.media.entity.ejb.EntityMediaBeanTester;
import org.jboss.test.media.entity.ejb.EntityMediaBeanTesterHome;

public class EntityMediaBeanUnitTestCase extends EJBTestCase
{
   private static final Logger log =
      Logger.getLogger(EntityMediaBeanUnitTestCase.class);

   // Attributes ----------------------------------------------------
   private EntityMediaBeanTesterHome home;
   private boolean resourcesLoaded;
   private String smallString;
   private String bigString;
   private byte[] smallBlob;
   private byte[] bigBlob;

   public static Test suite() throws Exception
   {
      return JBossTestCase.getDeploySetup(
         EntityMediaBeanUnitTestCase.class,
         "media-entity.jar");
   }

   public EntityMediaBeanUnitTestCase(String name) throws Exception
   {
      super(name);
   }

   public void testCreate() throws Exception
   {
      log.debug("testCreate");
      EntityMediaBeanTester tester = home.create();
      String id = tester.createEntityMediaBean();
      log.debug(" Created Entity Bean with id=" + id);
      tester.removeEntityMediaBean(id);
   }

   public void testContent() throws Exception
   {
      log.debug("testEmpty");
      EntityMediaBeanTester tester = home.create();
      String id = tester.createEntityMediaBean();

      try
      {
         byte[] nullContent = tester.getContent(id);
         fail("Should have thrown ContentAccessException");
      }
      catch (ContentAccessException e)
      {
         // expected
      }

      byte[] dummyContent = { '1', '2', '3', '4', '5' };
      tester.setContent(id, dummyContent);
      byte[] content = tester.getContent(id);
      assertEquals(dummyContent, content);
      tester.removeEntityMediaBean(id);
   }

   /**
    * Lookup the LOB lobHome and cache it.
    * Load the test data.
    */
   public void setUpEJB() throws Exception
   {
      log.debug("setupEJB");

      if (!resourcesLoaded)
      {
         Context initialContext = new InitialContext();
         Object ref = initialContext.lookup("ejb/EntityMediaBeanTester");
         home =
            (EntityMediaBeanTesterHome) PortableRemoteObject.narrow(
               ref,
               EntityMediaBeanTesterHome.class);

         resourcesLoaded = true;
      }
   }

   /**
    * Remove data references so that they can be garbage collected if needed.
    */
   public void tearDownEJB() throws Exception
   {
      log.debug("tearDownEJB");
   }

   // Protected -------------------------------------------------------

   static void assertEquals(byte[] expected, byte[] actual)
   {
      assertEquals(expected.length, actual.length);
      for (int i = 0; i < expected.length; ++i)
         assertEquals(expected[i], actual[i]);
   }

   // Private -------------------------------------------------------

   /**
    * Return the content of the input stream provided as a byte array.
    * @param   resourceName  resource to read
    * @return  content as a byte array
    */
   private static final byte[] loadBinaryData(String resourceName)
   {
      ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
      InputStream input = classLoader.getResourceAsStream(resourceName);
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      try
      {
         int byteRead;
         while ((byteRead = input.read()) != -1)
            baos.write(byteRead);
         return baos.toByteArray();
      }
      catch (Exception e)
      {
         throw new IllegalStateException(e.getMessage());
      }
      finally
      {
         try
         {
            baos.close();
         }
         catch (Exception e)
         {
         }
         try
         {
            input.close();
         }
         catch (Exception e)
         {
         }
      }
   }

   /**
    * Return the content of the input stream provided as a String.
    * @param   resourceName resource to read
    * @return  content as a string
    */
   private static final String loadTextData(String resourceName)
   {
      ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
      InputStream input = classLoader.getResourceAsStream(resourceName);
      StringWriter stringWriter = new StringWriter();
      try
      {
         int byteRead;
         while ((byteRead = input.read()) != -1)
            stringWriter.write(byteRead);
         return stringWriter.toString();
      }
      catch (Exception e)
      {
         throw new IllegalStateException(e.getMessage());
      }
      finally
      {
         try
         {
            stringWriter.close();
         }
         catch (Exception e)
         {
         }
         try
         {
            input.close();
         }
         catch (Exception e)
         {
         }
      }
   }
}