/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.test.compatibility.test;

import java.io.Externalizable;
import java.io.File;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import junit.framework.AssertionFailedError;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestResult;
import junit.framework.TestSuite;

import org.jboss.logging.Logger;

/**
 * Verifies if all classes are implementing serialVersionUID in a given JAR.
 * Use the method createCompatilibitySuite defined in this class embedded in suite methods. 
 * @author clebert.suconic@jboss.com
 */
public class TestDefinitionOfSerialVersionUID extends TestCase
{
    static Logger log = Logger.getLogger("TestDefinitionOfSerialVersionUID");


    static class TestLookupIndividualSerialUID extends TestCase
    {
        
        Class loadedClass;
        public TestLookupIndividualSerialUID(Class loadedClass)
        {
            super(loadedClass.getName());
            this.loadedClass=loadedClass;
        }

        public void run(TestResult result)
        {
            log.info("Testing " + this.getName());
            try
            {
                result.startTest(this);

                Field primaryField = lookupSerialField(loadedClass);
                assertNotNull("serialVersionUID not defined in current version of " + this.getName(), primaryField);
                primaryField.setAccessible(true);
                log.info("SerialUID for " + this.getName() + " is " + primaryField.getLong(null));
            } 
            catch (AssertionFailedError error)
            {
                result.addFailure(this, error);
            } catch (Throwable e)
            {
                result.addError(this,e);
            }
            log.info("Done " + this.getName());
            result.endTest(this);
        }

        /** Using reflection as ObjectSerialClass will calculate a new one */
        private Field lookupSerialField(Class parameterClass) throws NoSuchFieldException
        {
            try 
            {
                return parameterClass.getDeclaredField("serialVersionUID");
            }
            catch (Throwable e)
            {
                return null;
            }
        }

    }

    /** Create a compatibility suite for a give JAR file */
    public static Test createCompatilibitySuite(File jarFile) throws Exception
    {
        URLClassLoader urlClassLoader = new URLClassLoader(new URL[] {jarFile.toURL()},TestDefinitionOfSerialVersionUID.class.getClassLoader());
        TestSuite suite = new TestSuite();
        ZipFile zipFile = new ZipFile(jarFile);
        Enumeration enum = zipFile.entries();
        while (enum.hasMoreElements())
        {
            ZipEntry entry = (ZipEntry) enum.nextElement();
            if (!entry.isDirectory() && entry.getName().endsWith(".class"))
            {
                String classname = entry.getName().replace('/', '.').substring(
                        0, entry.getName().length() - 6);
                
                Class lookupClass = null;
                try {
                    lookupClass = urlClassLoader.loadClass(classname);
                }
                catch (Throwable e) 
                {
                    System.out.println("Warning... Class " + classname + " couldn't be loaded at " + jarFile.getName() + " -> " + e.toString());
                }
                
                
                if (lookupClass!=null && (Externalizable.class.isAssignableFrom(lookupClass) || Serializable.class.isAssignableFrom(lookupClass)) && !lookupClass.isInterface()) {
                    suite.addTest(new TestLookupIndividualSerialUID(lookupClass));
                }
            }
        }
        return suite;
    }
    
    public static Test createCompatilibitySuiteForLib(String libFileName) throws Exception
    {
        String jbossdist = System.getProperty("jboss.dist");
        if (jbossdist==null) {
                System.out.println("jboss.dist not defined");
                throw new Exception("jboss.dist not defined");
        }
        String strfile = jbossdist +  "/server/all/lib/" + libFileName;

        File file = new File(strfile);
        if (file.exists())
        {
           return createCompatilibitySuite(file);
        } else 
        {
           strfile = jbossdist + "/lib/" + libFileName;
           file = new File(strfile);
           if (file.exists()) 
           {
               return createCompatilibitySuite(file);
           } else 
           {
                   System.out.println("library " + strfile + " not found");
                   throw new RuntimeException ("library " + strfile + " not found");
           }
        }
        
    }

}
