/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.test.compatibility.test;

import java.io.FileInputStream;
import java.io.ObjectInputStream;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.Iterator;

import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.framework.Test;
import org.jboss.tools.SerialVersionUID;
import org.jboss.tools.ClassVersionInfo;


/** Tests of serial version uid compatibility across jboss versions
 *
 *  @author Scott.Stark@jboss.org
 *  @version $Revision:1$
 */
public class SerialVersionUIDUnitTestCase
      extends TestCase
{
   static Map currentClassInfoMap;

   public SerialVersionUIDUnitTestCase(String name)
   {
      super(name);
   }

   /** Validate the 4.0.1 serial version uids against the current build
    * @throws Exception
    */
   public void test401Compatibility()
         throws Exception
   {
      // The packages in jboss-4.0.x with known 4.0.1 serialization issues
      String[] badPackages = {
         // Ignore org.apache.* issues
         "org.apache",
         // Ignore jacorb packages org.jacorb.*, org.omg.*
         "org.jacorb",
         "org.omg",
         /* Ignore the aop package for 4.0.1 since this will be in flux until
            jboss5/ejb3 is stablized
         */
         "org.jboss.aop",
         /* Ignore org.jboss.webservice for 4.0.1 since the org.apache.axis to
            org.jboss.axis package name change breaks serialization
         */
         "org.jboss.webservice",
         /* Ignore org.hsqldb as there are some utility classes that changed
         in the upgrade to 1_8_0
         */
         "org.hsqldb"
      };

      System.out.println("+++ test401Compatibility");
      // load the 4.0.1 serialVersionUID database
      String etc = System.getProperty("jbosstest.src.etc", "../src/etc");
      File serFile = new File(etc, "serialVersionUID/401.ser");
      FileInputStream fis = new FileInputStream(serFile);
      ObjectInputStream ois = new ObjectInputStream(fis);
      Map classInfoMap = (Map) ois.readObject();
      System.out.println("4.0.1 serial classes count: "+classInfoMap.size());

      System.setProperty("org.jboss.j2ee.LegacySerialization", "true");
      Map currentClassInfoMap = calcClassInfoMap();
      int mismatchCount = compare(classInfoMap, currentClassInfoMap, "401", badPackages);
      currentClassInfoMap.clear();
      System.out.println("serialVersionUID mismatches = "+mismatchCount);
      assertTrue("There are no serialVersionUID mismatches("+mismatchCount+")",
         mismatchCount == 0);
   }

   /** Validate the J2EE 1.4.1 RI serial version uids against the current build
    * @throws Exception
    */
   public void testJ2EERI141Compatibility()
         throws Exception
   {
      // The packages in j2ee 1.4.1RI with known serialization issues
      String[] badPackages = {
         // The javax.mail binaries in the ri are not consistent with the javamail 1.3FCS
         "javax.mail"
      };
      System.out.println("+++ testJ2EERI141Compatibility");
      System.getProperties().remove("org.jboss.j2ee.LegacySerialization");
      String etc = System.getProperty("jbosstest.src.etc", "../src/etc");
      File serFile = new File(etc, "serialVersionUID/j2ee141.ser");
      FileInputStream fis = new FileInputStream(serFile);
      ObjectInputStream ois = new ObjectInputStream(fis);
      Map classInfoMap = (Map) ois.readObject();
      System.out.println("J2EE RI serial classes count: "+classInfoMap.size());

      Map currentClassInfoMap = calcClassInfoMap();
      int mismatchCount = compare(classInfoMap, currentClassInfoMap, "J2EE1.4", badPackages);
      currentClassInfoMap.clear();
      System.out.println("serialVersionUID mismatches = "+mismatchCount);
      assertTrue("There are no serialVersionUID mismatches("+mismatchCount+")",
         mismatchCount == 0);
   }

   private int compare(Map classInfoMap, Map currentClassInfoMap,
      String versionName, String[] badPackages)
   {
      int mismatchCount = 0;
      Iterator iter = currentClassInfoMap.values().iterator();
      while( iter.hasNext() )
      {
         ClassVersionInfo cvi = (ClassVersionInfo) iter.next();
         String name = cvi.getName();
         ClassVersionInfo cvi401 = (ClassVersionInfo) classInfoMap.get(name);
         if( cvi401 != null && cvi.getSerialVersion() != cvi401.getSerialVersion() )
         {
            String msg = "serialVersionUID error for "+name
               +", " + versionName + " " + cvi401.getSerialVersion()
               +", current: "+cvi.getSerialVersion();
            // Don't count classes from badPackages
            boolean isInBadPkg = false;
            for(int n = 0; n < badPackages.length; n ++)
            {
               String pkg = badPackages[n];
               if( name.startsWith(pkg) )
               {
                  isInBadPkg = true;
                  break;
               }
            }
            if( isInBadPkg == false )
            {
               mismatchCount ++;
               System.err.println(msg);
            }
            else
            {
               System.out.println(msg);               
            }
         }
      }
      return mismatchCount;
   }

   static Map calcClassInfoMap()
      throws IOException
   {
      String jbossDist = System.getProperty("jbosstest.dist");
      File jbossHome = new File(jbossDist);
      jbossHome = jbossHome.getCanonicalFile();
      System.out.println("Calculating serialVersionUIDs for jbossHome: "+jbossHome);
      Map classInfoMap = SerialVersionUID.generateJBossSerialVersionUIDReport(
         jbossHome);
      return classInfoMap;
   }
   public static Test suite() throws Exception
   {
      TestSuite suite = new TestSuite();
      suite.addTest(new TestSuite(SerialVersionUIDUnitTestCase.class));
      return suite;
   }
}
