/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.system.filterfactory;

import javax.management.AttributeChangeNotificationFilter;
import javax.management.NotificationFilter;

import org.jboss.system.NotificationFilterFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Factory for AttributeChangeNotificationFilter filters.
 * 
 * The produced filter filters-in AttributeChangeNotifications 
 * for zero or more attributes, so you need to explicitly enable
 * the attribute names you are interested in.
 *
 * The passed filterConfig xml element fragment should look like:
 * 
 * <filter factory="AttributeChangeNotificationFilterFactory">
 *   <enable attribute-name="State"/>
 *   ...
 * </filter>
 * 
 * @author <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 * @version $Revision:1$
**/
public class AttributeChangeNotificationFilterFactory
   implements NotificationFilterFactory
{
   // Constants -----------------------------------------------------
   
   /** the xml element and attribute supported by this factory */
   public static final String ENABLE_ELEMENT = "enable";
   public static final String ENABLE_ATTRNAME_ATTRIBUTE = "attribute-name";
   
   /**
    * Default public CTOR (necessary)
    */
   public AttributeChangeNotificationFilterFactory()
   {
      // empty
   }
   
   /**
    * The actual filter factory implementation
    */
   public NotificationFilter createNotificationFilter(Element filterConfig)
      throws Exception
   {
      // start off with a filter that does not allow any named attribute
      AttributeChangeNotificationFilter filter = new AttributeChangeNotificationFilter();
      
      // filterConfig should point to the <filter factory="..."> element,
      // we are interested in its 'enable' children to configure the filter 
      NodeList filterChildren = filterConfig.getChildNodes();
      
      for (int i = 0; i < filterChildren.getLength(); i++) 
      {
         Node filterChildNode = filterChildren.item(i);
      
         // check if this is an 'enable' element, ignore everything else
         if (filterChildNode.getNodeName().equals(ENABLE_ELEMENT)) 
         {
            // look for 'attribute-name' attribute
            if (((Element)filterChildNode).hasAttribute(ENABLE_ATTRNAME_ATTRIBUTE)) 
            {
               String attributeName = ((Element)filterChildNode).getAttribute(ENABLE_ATTRNAME_ATTRIBUTE);
               // enable this type in the filter
               filter.enableAttribute(attributeName);
            }
            else
            {
               throw new Exception("'" + ENABLE_ELEMENT + "' element must have a '"
                     + ENABLE_ATTRNAME_ATTRIBUTE + "' attribute");
            }
         }
      }
      // we are done
      return filter;
   }
}
