/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting.performance.standard;

import org.apache.log4j.Level;
import org.jboss.dtf.MultipleTestRunner;
import org.jboss.remoting.oneway.OnewayInvokerClientTest;
import org.jboss.remoting.performance.PerformanceReporter;
import org.jboss.remoting.performance.PerformanceTest;

import java.util.HashMap;
import java.util.Map;

/**
 * Makes oneway call a number of times using multiple threads
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class StandardTest extends OnewayInvokerClientTest
{

   private final Object waitObj = new Object();
   private int callCount = 0;

   public StandardTest(String name)
   {
      super(name);
   }

   public StandardTest(int numberOfInstances)
   {
      super(numberOfInstances);
   }

   public StandardTest(String transport, int port)
   {
      super(transport, port);
   }

   public StandardTest(String transport, int port, int numberOfInstances)
   {
      super(transport, port, numberOfInstances);
   }

   protected int incrementCallCount()
   {
      int currentCount = callCount++;
      if((currentCount % 100) == 0)
      {
         System.out.println("call count: " + currentCount);
      }
      if(callCount == (PerformanceTest.NUM_OF_THREADS * PerformanceTest.NUM_OF_CALLS))
      {
         synchronized(waitObj)
         {
            waitObj.notify();
         }
      }
      return currentCount;
   }

   protected void setCallCount(int count)
   {
      this.callCount = count;
   }

//   protected void makeClientOnewayInvocation(String method, String param) throws Throwable
//   {
//      makeOnewayInvocation(method, param);
//   }
//
//   protected void makeServerOnewayInvocation(String method, String param) throws Throwable
//   {
//      makeOnewayInvocation(method, param);
//   }
//
//   protected void superMakeClientOnewayInvocation(String method, String param) throws Throwable
//   {
//      super.makeClientOnewayInvocation(method, param);
//   }
//
//   protected void superMakeServerOnewayInvocation(String method, String param) throws Throwable
//   {
//      super.makeServerOnewayInvocation(method, param);
//   }

   protected Object superMakeClientInvocation(String method, String param) throws Throwable
   {
      return super.makeInvocation(method, param);
   }

//   protected void makeOnewayInvocation(String method, String param) throws Throwable
   protected Object makeClientInvocation(String method, String param) throws Throwable
   {
      callCount = 0;

      sendTotalCount();

      printStartMessage();

      long startTime = System.currentTimeMillis();

      System.out.println("Start time: " + startTime);

      for(int x = 0; x < PerformanceTest.NUM_OF_THREADS; x++)
      {
         if((PerformanceTest.NUM_OF_THREADS % 10) == 0)
         {
            System.out.println("started " + x + " threads");
         }
         new Thread(getRunner(method)).start();
      }

      synchronized(waitObj)
      {
         try
         {
            waitObj.wait(5 * 60 * 1000); // timeout if not notified
         }
         catch(InterruptedException e)
         {

         }
      }

      long endTime = System.currentTimeMillis();

      System.out.println("End time: " + endTime);
      System.out.println("Total number of calls: " + callCount);
      long totalTime = endTime - startTime;
      System.out.println("Total time: " + totalTime);

      int svrTotalCount = getServerTotalCount();

      System.out.println("Server total count: " + svrTotalCount);
      assertEquals(callCount, svrTotalCount);

      Map metadata = new HashMap();
      metadata.put("transport", getTransport());
      metadata.put("server total count", String.valueOf(svrTotalCount));
      metadata.put("number of client threads", String.valueOf(PerformanceTest.NUM_OF_THREADS));
      metadata.put("number of client calls per thread", String.valueOf(PerformanceTest.NUM_OF_CALLS));


      PerformanceReporter.writeReport(this.getClass().getName() + "::" + method,
                                      totalTime, callCount, metadata);

      return null;
   }

   protected void checkAssertion(String param, Object obj)
   {
      // NO OP - over ride from OnewayInvokerClientTest so don't get failures.
   }


   private int getServerTotalCount()
   {
      int svrCount = 0;

      try
      {
         Object ret = makeInvocation("serverTotalCallCount", null);
         if(ret != null && ret instanceof String)
         {
            svrCount = Integer.parseInt((String) ret);
         }
      }
      catch(Throwable throwable)
      {
         throwable.printStackTrace();
      }
      return svrCount;
   }

   protected void printStartMessage()
   {
      System.out.println("\n*****************************************************\n" +
                         "Starting standard preformance test with client.\n" +
                         "*****************************************************\n");
   }

   /**
    * Used to tell the test server handler the number of calls to expect.
    */
   private void sendTotalCount() throws Throwable
   {
      makeInvocation("totalCallCount", String.valueOf(PerformanceTest.NUM_OF_CALLS * PerformanceTest.NUM_OF_THREADS));
   }

   protected Runnable getRunner(String method)
   {
      return new ClientRunner(method);
   }

   public void testOnewayServerInvocation() throws Throwable
   {
      //NO OP since don't want to run this one in this case
   }

   public void testOnewayClientInvocation() throws Throwable
   {
      //NO OP since don't want to run this one in this case
   }


   public class ClientRunner implements Runnable
   {
      private String method;

      public ClientRunner(String method)
      {
         this.method = method;
      }

      public void run()
      {
         for(int i = 0; i < PerformanceTest.NUM_OF_CALLS; i++)
         {
            try
            {
               superMakeClientInvocation(this.method, String.valueOf(incrementCallCount()));
            }
            catch(Throwable throwable)
            {
               throwable.printStackTrace();
            }
         }
      }

   }

   public static void main(String[] args)
   {

      org.apache.log4j.BasicConfigurator.configure();
      org.apache.log4j.Category.getRoot().setLevel(Level.INFO);
      org.apache.log4j.Category.getInstance("org.jgroups").setLevel(Level.INFO);
      //org.apache.log4j.Category.getInstance("org.jboss.remoting").setLevel(Level.DEBUG);
      //org.apache.log4j.Category.getInstance("test").setLevel(Level.DEBUG);

      StandardTest client = null;
      if(args.length == 1)
      {
         int instances = Integer.parseInt(args[0]);
         client = new StandardTest(instances);
      }
      else if(args.length == 2)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         client = new StandardTest(transport, port);
      }
      else if(args.length == 3)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         int instances = Integer.parseInt(args[2]);
         client = new StandardTest(transport, port, instances);
      }
      else
      {
         client = new StandardTest(OnewayInvokerClientTest.class.getName());
         System.out.println("Using default transport (" + client.getTransport() +
                            ") and default port (" + client.getPort() + ") and " +
                            "default number of instances (" + client.getNumberOfInstances() + ")" +
                            "\nCan enter transport, port, and instances via command line.");
      }

      try
      {
         //regular class run
         //client.runInvokers();
         MultipleTestRunner runner = new MultipleTestRunner();
         runner.doRun(client, true);
      }
      catch(Throwable e)
      {
         e.printStackTrace();
         System.exit(1);
      }
      System.exit(0);
   }


}