/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting.callback.pull.memory;

import org.jboss.remoting.Callback;
import org.jboss.remoting.HandleCallbackException;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerCallbackHandler;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.ServerInvokerCallbackHandler;

import javax.management.MBeanServer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @author <a href="mailto:tom@jboss.org">Tom Elrod</a>
 */
public class CallbackInvocationHandler implements ServerInvocationHandler, Runnable
{
   private transient List pullListeners = new ArrayList();
   private transient List pushListeners = new ArrayList();

   private int numberOfCallbacks = 500;

   private boolean isDone = false;

   private int callbackCounter = 0;

   private byte[] memHolder = null;

   public CallbackInvocationHandler()
   {
      long max = Runtime.getRuntime().maxMemory();
      System.out.println("max mem: " + max);
      int memSize = (int) (max * 0.7);
      System.out.println("70% of max: " + memSize);
      long free = Runtime.getRuntime().freeMemory();
      System.out.println("free mem: " + free);
      long total = Runtime.getRuntime().totalMemory();
      System.out.println("total mem: " + total);
      if(total != max)
      {
         memHolder = new byte[memSize];
      }
      else if(free > memSize)
      {
         memHolder = new byte[memSize];
      }
   }

   /**
    * called to handle a specific invocation
    *
    * @param invocation
    * @return
    * @throws Throwable
    */
   public Object invoke(InvocationRequest invocation) throws Throwable
   {
      if("getdone".equalsIgnoreCase((String) invocation.getParameter()))
      {
         return new Boolean(isDone);
      }

      try
      {
         numberOfCallbacks = Integer.parseInt((String) invocation.getParameter());
      }
      catch(NumberFormatException e)
      {
         new Thread(this).start();
         return "Starting callback";
      }

      return null;
   }

   /**
    * When an object implementing interface <code>Runnable</code> is used to create a thread, starting the thread causes
    * the object's <code>run</code> method to be called in that separately executing thread.
    * <p/>
    * The general contract of the method <code>run</code> is that it may take any action whatsoever.
    *
    * @see Thread#run()
    */
   public void run()
   {

      try
      {
         synchronized(pullListeners)
         {
            for(int x = 0; x < numberOfCallbacks; x++)
            {
               if(x % 10 == 0)
               {
                  System.out.println("x = " + x);
                  System.out.println("Free mem = " + Runtime.getRuntime().freeMemory());
               }
               // Will also fire callback to listeners if they were to exist using
               // simple invocation request.
               synchronized(pullListeners)
               {
                  Iterator itr = pullListeners.iterator();
                  while(itr.hasNext())
                  {
                     InvokerCallbackHandler callbackHandler = (InvokerCallbackHandler) itr.next();
                     try
                     {
                        callbackHandler.handleCallback(new Callback(getCallbackMessage()));
                        if(isMemLow())
                        {
                           Thread.currentThread().sleep(1000);
                        }
                     }
                     catch(HandleCallbackException e)
                     {
                        e.printStackTrace();
                     }
                  }
               }
            }
            // done adding callbacks, now release memory
            memHolder = null;
         }

         isDone = true;

         synchronized(pushListeners)
         {
            Iterator itr = pushListeners.iterator();
            while(itr.hasNext())
            {
               InvokerCallbackHandler handler = (InvokerCallbackHandler) itr.next();
               try
               {
                  handler.handleCallback(new Callback("Done"));
               }
               catch(HandleCallbackException e)
               {
                  e.printStackTrace();
               }
            }
         }
      }
      catch(Throwable e)
      {
         e.printStackTrace();
      }

   }

   private boolean isMemLow()
   {
      Runtime runtime = Runtime.getRuntime();
      long max = runtime.maxMemory();
      long total = runtime.totalMemory();
      long free = runtime.freeMemory();
      float percentage = 100 * free / total;
      if(max == total && 40 >= percentage)
      {
         return true;
      }
      else
      {
         return false;
      }
   }


   /**
    * Adds a callback handler that will listen for callbacks from the server invoker handler.
    *
    * @param callbackHandler
    */
   public void addListener(InvokerCallbackHandler callbackHandler)
   {
      ServerInvokerCallbackHandler sih = (ServerInvokerCallbackHandler) callbackHandler;

      if(!sih.isPullCallbackHandler())
      {
         pushListeners.add(callbackHandler);
      }
      else
      {
         pullListeners.add(callbackHandler);
      }
   }

   /**
    * Removes the callback handler that was listening for callbacks from the server invoker handler.
    *
    * @param callbackHandler
    */
   public void removeListener(InvokerCallbackHandler callbackHandler)
   {
      pullListeners.remove(callbackHandler);
      pushListeners.remove(callbackHandler);
   }

   /**
    * set the mbean server that the handler can reference
    *
    * @param server
    */
   public void setMBeanServer(MBeanServer server)
   {
      // NO OP as do not need reference to MBeanServer for this handler
   }

   /**
    * set the invoker that owns this handler
    *
    * @param invoker
    */
   public void setInvoker(ServerInvoker invoker)
   {
      // NO OP as do not need reference back to the server invoker
   }

   private Object getCallbackMessage()
   {
      callbackCounter++;
      //byte[] bytes = new byte[5120000];
      byte[] bytes = new byte[102400];
      TestCallback callback = new TestCallback(bytes, callbackCounter);
      return callback;
   }


}