/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
package org.jboss.remoting;

import org.jboss.logging.Logger;
import org.jboss.remoting.loading.ClassByteClassLoader;
import org.jboss.remoting.marshal.MarshallLoaderFactory;

/**
 * AbstractInvoker is an abstract handler part that contains common methods between both
 * client and server.
 *
 * @author <a href="mailto:jhaynie@vocalocity.net">Jeff Haynie</a>
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 * @version $Revision:1$
 */
public abstract class AbstractInvoker implements Invoker
{
    protected final Logger log = Logger.getLogger(getClass());
    protected ClassByteClassLoader classbyteloader;
    protected InvokerLocator locator;
    protected InvokerLocator localServerLocator;

    public AbstractInvoker (InvokerLocator locator)
    {
        this.classbyteloader = new ClassByteClassLoader(this.getClass().getClassLoader());
        this.locator = locator;

        localServerLocator = InvokerRegistry.getSuitableServerLocatorForRemote(locator);
        try
        {
           InvokerLocator loaderLocator = MarshallLoaderFactory.convertLocator(locator);
           if(loaderLocator != null)
           {
             classbyteloader.setClientInvoker(new Client(loaderLocator));
           }
        }
        catch(Exception e)
        {
           log.error("Could not create remote class loading for invoker.", e);
        }
    }

    /**
     * return the locator this Invoker represents
     *
     * @return
     */
    public InvokerLocator getLocator ()
    {
        return locator;
    }

    /**
     * This should be set when want to override the default behavior of automatically
     * getting s suitable locator.  This should be used want want to control what type
     * of callbacks to receive (pull or push).  Set to null to poll for callback messages.
     * This can also be used to receive callbacks using another transport and subsystem,
     * if desired.
     * @param locator
     */
    public void setClientLocator(InvokerLocator locator)
    {
        localServerLocator = locator;
    }

    /**
     * Gets the client locator.  This locator will be used by the server side
     * to make callbacks to the handler for this locator.
     * @return
     */
    public InvokerLocator getClientLocator()
    {
        return localServerLocator;
    }

    /**
     * set the classloader to use internally
     *
     * @param classloader
     */
    public synchronized void setClassLoader (ClassLoader classloader)
    {
        this.classbyteloader = new ClassByteClassLoader(classloader);
    }

   public synchronized ClassLoader getClassLoader()
   {
      return classbyteloader;
   }

}
