/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.media.format.image.iio;

import javax.imageio.metadata.IIOMetadata;

/**
 * PNG <code>MediaHeader</code> based on the Java Image I/O API.
 *
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:ricardoarguello@users.sourceforge.net">Ricardo Argello</a>
 */
public class PNGMediaHeader extends IIOMediaHeader
{
   // FIXME
   private static final long serialVersionUID = 0L;

   private static final String PNG_METADATA_FORMAT_NAME =
      "javax_imageio_png_1.0";

   private static final String[] FIELD_NAMES =
      {
         "width",
         "height",
         "bitDepth",
         "colorType",
         "compressionMethod",
         "filterMethod",
         "interlaceMethod" };

   /** The IHDR chunk, containing the header. */
   private static final String IHDR = "IHDR";

   /**
    * Constructor.
    *
    * @param metadata
    * @param metadataFormat
    */
   public PNGMediaHeader(IIOMetadata imageMetadata)
   {
      super(imageMetadata, PNG_METADATA_FORMAT_NAME);
   }

   /**
    * @see javax.emb.MediaHeader#getFieldNames()
    */
   public String[] getFieldNames()
   {
      return FIELD_NAMES;
   }

   /**
    * @see javax.emb.MediaHeader#getField(java.lang.String)
    */
   public Object getField(String fieldname)
   {
      Object field = null;

      if (fieldname.equals("width"))
      {
         field = new Integer(getWidth());
      }

      if (fieldname.equals("height"))
      {
         field = new Integer(getHeight());
      }

      if (fieldname.equals("bitDepth"))
      {
         field = getBitDepth();
      }

      if (fieldname.equals("colorType"))
      {
         field = getColorType();
      }

      if (fieldname.equals("compressionMethod"))
      {
         field = getCompressionMethod();
      }

      if (fieldname.equals("filterMethod"))
      {
         field = getFilterMethod();
      }

      if (fieldname.equals("interlaceMethod"))
      {
         field = getInterlaceMethod();
      }

      return field;
   }

   /**
    * The width of the image in pixels.
    *
    * @return 1 (inclusive) to 2147483647 (inclusive).
    */
   public int getWidth()
   {
      String width = getAttribute(IHDR, "width");
      return Integer.parseInt(width);
   }

   /**
    * The height of the image in pixels.
    *
    * @return 1 (inclusive) to 2147483647 (inclusive).
    */
   public int getHeight()
   {
      String height = getAttribute(IHDR, "height");
      return Integer.parseInt(height);
   }

   /**
    * The bit depth of the image samples.
    *
    * @return "1", "2", "4", "8" or "16".
    */
   public String getBitDepth()
   {
      return getAttribute(IHDR, "bitDepth");
   }

   /**
    * The color type of the image.
    *
    * @return "Grayscale", "RGB", "Palette", "GrayAlpha" or "RGBAlpha".
    */
   public String getColorType()
   {
      return getAttribute(IHDR, "colorType");
   }

   /**
    * The compression used for image data.
    *
    * @return allways "deflate".
    */
   public String getCompressionMethod()
   {
      return getAttribute(IHDR, "compressionMethod");
   }

   /**
    * The filtering method used for compression.
    *
    * @return allways "adaptive".
    */
   public String getFilterMethod()
   {
      return getAttribute(IHDR, "filterMethod");
   }

   /**
    * The interlacing method.
    *
    * @return "none" or "adam7".
    */
   public String getInterlaceMethod()
   {
      return getAttribute(IHDR, "interlaceMethod");
   }

   /**
    * The image gamma, multiplied by 1e5.
    *
    * @return 0 (inclusive) to 2147483647 (inclusive).
    */
   public int getGamma()
   {
      String gamma = getAttribute("gAMA", "value");
      return Integer.parseInt(gamma);
   }

   /**
    * The number of horizontal pixels per unit, multiplied by 1e5.
    *
    * @return 0 (inclusive) to 2147483647 (inclusive).
    */
   public int getPixelsPerUnitXAxis()
   {
      String pixelsPerUnitXAxis = getAttribute("pHYS", "pixelsPerUnitXAxis");
      return Integer.parseInt(pixelsPerUnitXAxis);
   }

   /**
    * The number of vertical pixels per unit, multiplied by 1e5.
    *
    * @return 0 (inclusive) to 2147483647 (inclusive).
    */
   public int getPixelsPerUnitYAxis()
   {
      String pixelsPerUnitYAxis = getAttribute("pHYS", "pixelsPerUnitYAxis");
      return Integer.parseInt(pixelsPerUnitYAxis);
   }

   /**
    * The unit specifier for this chunk (i.e., meters).
    *
    * @return "unknown" or "meter".
    */
   public String getPixelsPerUnitSpecifier()
   {
      return getAttribute("pHYS", "unitSpecifier");
   }
}
