/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.management.j2ee;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Root class of the JBoss JSR-77 implementation of
 *
 * @author <a href="mailto:andreas@jboss.org">Andreas Schaefer</a>.
 * @author Scott.Stark@jboss.org
 * @author thomas.diesler@jboss.org
 * @version $Revision:1$
 * @jmx:mbean extends="org.jboss.management.j2ee.J2EEManagedObjectMBean"
 */
public class J2EEDomain
        extends J2EEManagedObject
        implements J2EEDomainMBean
{

   // Attributes ----------------------------------------------------
   /**
    * The local server J2EEDomain implementation name
    */
   private static String domainName = null;

   /**
    * list of servers associated with the domain as strings
    */
   private List serverNames = new ArrayList();

   // Static --------------------------------------------------------
   /**
    * Get the local J2EEDomain instance name
    *
    * @return the J2EEDomain object name for the local server.
    */
   public static String getDomainName()
   {
      return domainName;
   }

   /**
    * Query for the J2EEServer MBean in the given domain.
    *
    * @param mbeanServer the local MBeanServer
    * @return the J2EEServer name if found, null otherwise
    */
   public static ObjectName getDomainServerName(MBeanServer mbeanServer)
   {
      ObjectName domainServer = null;
      try
      {
         // Query for all MBeans matching the J2EEServer naming convention
         ObjectName serverQuery = new ObjectName(domainName + ":" +
                 J2EEManagedObject.TYPE + "=" + J2EETypeConstants.J2EEServer + "," + "*");

         Set servers = mbeanServer.queryNames(serverQuery, null);
         if (servers.isEmpty() == false)
         {
            domainServer = (ObjectName) servers.iterator().next();
         }
      }
      catch (Exception ignore)
      {
      }
      return domainServer;
   }

   // Constructors --------------------------------------------------

   public J2EEDomain(String domainName)
           throws MalformedObjectNameException,
           InvalidParentException
   {
      super(domainName, J2EETypeConstants.J2EEDomain, "Manager");
      J2EEDomain.domainName = domainName;
   }

   // Public --------------------------------------------------------

   /**
    * Return the J2EEServer names associated with this domain.
    *
    * @jmx:managed-attribute
    */
   public String[] getservers()
   {
      String[] servers = new String[serverNames.size()];
      serverNames.toArray(servers);
      return servers;
   }

   /**
    * @jmx:managed-operation
    */
   public String getserver(int pIndex)
   {
      if (pIndex >= 0 && pIndex < serverNames.size())
      {
         return (String) serverNames.get(pIndex);
      }
      return null;
   }

   // J2EEManagedObject implementation ----------------------------------------------

   public void addChild(ObjectName pChild)
   {
      String lType = J2EEManagedObject.getType(pChild);
      if (J2EETypeConstants.J2EEServer.equals(lType))
      {
         serverNames.add(pChild.getCanonicalName());
      }
   }

   public void removeChild(ObjectName pChild)
   {
      String lType = J2EEManagedObject.getType(pChild);
      if (J2EETypeConstants.J2EEServer.equals(lType))
      {
         serverNames.remove(pChild.getCanonicalName());
      }
   }
   // Object overrides ---------------------------------------------------

   public String toString()
   {
      return "J2EEDomain { " + super.toString() + " } [ " +
              ", servers: " + serverNames +
              " ]";
   }

   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------
}
