/**
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.mx.server;

// $Id:ExceptionHandler.java,v 1.0, 2005-06-24 19:04:00Z, Robert Worsnop$

import javax.management.*;

/**
 * Handles exceptions and wraps them if neccessary, arccording to the spec.
 * 
 * @author thomas.diesler@jboss.org
 */
public class ExceptionHandler
{

   // hide constructor
   private ExceptionHandler()
   {
   }

   /**
    * Handles exceptions and wraps them if neccessary, arccording to the spec.
    *
    * @param t the exception thrown by the invocation
    * @return any wrapped exception
    */
   public static JMException handleException(Throwable t)
   {
      handleRuntimeExceptionOrError(t);

      // when we get here, only exceptions are left
      Exception e = (Exception)t;

      if (e instanceof OperationsException)
         return (OperationsException)e;
      if (e instanceof ReflectionException)
         return (ReflectionException)e;
      if (e instanceof MBeanRegistrationException)
         return (MBeanRegistrationException)e;

      // wrap the core java exceptions
      if (e instanceof ClassNotFoundException)
         return new ReflectionException(e);
      if (e instanceof IllegalAccessException)
         return new ReflectionException(e);
      if (e instanceof InstantiationException)
         return new ReflectionException(e);
      if (e instanceof NoSuchMethodException)
         return new ReflectionException(e);

      // The MBeanException is the one that might wrap other exceptions
      // For example, the AbstractMBeanInvoker.invoke cannot throw OperationsException
      if (e instanceof MBeanException)
      {
         Throwable cause = e.getCause();

         if (cause instanceof JMException)
            return (JMException)cause;
         else
            return (MBeanException)e;
      }

      // wrap any exception thrown by an mbean
      return new MBeanException(e);
   }

   /**
    * Handles runtime exceptions and rethrows them wraped if neccessary, arccording to the spec.
    *
    * @param e the exception thrown by the invocation
    */
   private static void handleRuntimeExceptionOrError(Throwable e)
   {
      // is already of throwable type
      if (e instanceof RuntimeOperationsException)
         throw (RuntimeOperationsException)e;
      if (e instanceof RuntimeErrorException)
         throw (RuntimeErrorException)e;
      if (e instanceof RuntimeMBeanException)
         throw (RuntimeMBeanException)e;

      // wrap java core runtime exceptions
      if (e instanceof IllegalArgumentException)
         throw new RuntimeOperationsException((IllegalArgumentException)e);
      if (e instanceof IndexOutOfBoundsException)
         throw new RuntimeOperationsException((IndexOutOfBoundsException)e);
      if (e instanceof NullPointerException)
         throw new RuntimeOperationsException((NullPointerException)e);

      // wrap any error
      if (e instanceof Error)
         throw new RuntimeErrorException((Error)e);

      // wrap any runtime exception
      if (e instanceof RuntimeException)
         throw new RuntimeMBeanException((RuntimeException)e);
   }

}
