/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
package org.jboss.mx.remoting;

import java.util.Collections;
import java.util.Map;
import java.util.HashMap;
import javax.management.MBeanServer;

import org.jboss.logging.Logger;
import org.jboss.mx.remoting.MBeanServerClientInvokerProxy;
import org.jboss.remoting.InvokerLocator;

/**
 * MBeanServerRegistry is a registry for remote MBeanServer proxies that are registered in the local
 * JVM.
 * 
 * @author <a href="mailto:jhaynie@vocalocity.net">Jeff Haynie</a>
 * @version $Revision:1$
 */
public class MBeanServerRegistry 
{
    private static final Logger log = Logger.getLogger(MBeanServerRegistry.class.getName());

    private static final Map serversById=Collections.synchronizedMap(new HashMap());
    private static final Map serversByLocator=Collections.synchronizedMap(new HashMap());

    /**
     * return true if the remote MbeanServer is registered for the given MBeanServer ID
     *
     * @param id
     * @return
     */
    public static final boolean isMBeanServerRegistered (String id)
    {
        return getMBeanServerFor(id)!=null;
    }
    /**
     * return the MBeanServer proxy for a given MBeanServer id, or null if none registered
     *
     * @param id
     * @return
     */
    public static final MBeanServer getMBeanServerFor (String id)
    {
        return (MBeanServer)serversById.get(id);
    }
    /**
     * returns true if the remote MBeanServer proxy is registered for the given locator
     *
     * @param locator
     * @return
     */
    public static final boolean isMBeanServerRegistered (InvokerLocator locator)
    {
        return getMBeanServerFor(locator)!=null;
    }
    /**
     * return the MBeanServer proxy for a given InvokerLocator, or null if none registered
     *
     * @param locator
     * @return
     */
    public static final MBeanServer getMBeanServerFor (InvokerLocator locator)
    {
        return (MBeanServer)serversByLocator.get(locator);
    }
    static synchronized void register (MBeanServer server, MBeanServerClientInvokerProxy proxy)
    {
       String serverid = proxy.getServerId();
       serversById.put(serverid, server);
       serversByLocator.put(proxy.getLocator(),server);
        if (log.isDebugEnabled())
        {
           log.debug("register called with proxy: "+proxy+" and serverid: "+serverid);
        }
    }
    static synchronized void unregister (MBeanServerClientInvokerProxy proxy)
    {
       String serverid = proxy.getServerId();
       serversById.remove(serverid);
       serversByLocator.remove(proxy.getLocator());
       if (log.isDebugEnabled())
       {
          log.debug("unregister called with proxy: "+proxy+" and serverid: "+serverid);
       }
    }
}
