/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.jboss.axis.encoding.ser;

import org.jboss.axis.description.TypeDesc;
import org.jboss.axis.encoding.Deserializer;
import org.jboss.axis.utils.BeanPropertyDescriptor;
import org.jboss.axis.utils.BeanUtils;
import org.jboss.axis.utils.JavaUtils;

import javax.xml.namespace.QName;
import java.util.HashMap;
import java.util.Map;

/**
 * DeserializerFactory for Bean
 *
 * @author Rich Scheuerle <scheu@us.ibm.com>
 * @author Sam Ruby <rubys@us.ibm.com>
 */
public class BeanDeserializerFactory extends BaseDeserializerFactory
{

   /**
    * Type metadata about this class for XML deserialization
    */
   protected TypeDesc typeDesc = null;
   protected Map propertyMap = null;

   public BeanDeserializerFactory(Class javaType, QName xmlType)
   {
      super(BeanDeserializer.class, xmlType, javaType);

      // Sometimes an Enumeration class is registered as a Bean.
      // If this is the case, silently switch to the EnumDeserializer
      if (JavaUtils.isEnumClass(javaType))
      {
         deserClass = EnumDeserializer.class;
      }

      typeDesc = TypeDesc.getTypeDescForClass(javaType);
      propertyMap = getProperties(javaType, typeDesc);
   }

   /**
    * Get a list of the bean properties
    */
   public static Map getProperties(Class javaType, TypeDesc typeDesc)
   {
      Map propertyMap = null;

      if (typeDesc != null)
      {
         propertyMap = typeDesc.getPropertyDescriptorMap();
      }
      else
      {
         BeanPropertyDescriptor[] pd = BeanUtils.getPd(javaType, null);
         propertyMap = new HashMap();
         // loop through properties and grab the names for later
         for (int i = 0; i < pd.length; i++)
         {
            BeanPropertyDescriptor descriptor = pd[i];
            propertyMap.put(descriptor.getName(), descriptor);
         }
      }

      return propertyMap;
   }

   /**
    * Optimize construction of a BeanDeserializer by caching the
    * type descriptor and property map.
    */
   protected Deserializer getGeneralPurpose(String mechanismType)
   {
      if (javaType == null || xmlType == null)
      {
         return super.getGeneralPurpose(mechanismType);
      }

      if (deserClass == EnumDeserializer.class)
      {
         return super.getGeneralPurpose(mechanismType);
      }

      return new BeanDeserializer(javaType, xmlType, typeDesc, propertyMap);
   }
}
