/*
 * JBoss, the OpenSource EJB server
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package javax.xml.rpc.soap;

import java.io.ObjectStreamField;
import java.io.ObjectInputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import javax.xml.soap.Detail;
import javax.xml.namespace.QName;

import org.jboss.util.id.SerialVersion;

/** The SOAPFaultException exception represents a SOAP fault.
 * 
 * The message part in the SOAP fault maps to the contents of faultdetail
 * element accessible through the getDetail method on the SOAPFaultException.
 * The method createDetail on the javax.xml.soap.SOAPFactory creates an instance
 * of the javax.xml.soap.Detail.
 * 
 * The faultstring provides a human-readable description of the SOAP fault. The
 * faultcode element provides an algorithmic mapping of the SOAP fault.
 *  
 * Refer to SOAP 1.1 and WSDL 1.1 specifications for more details of the SOAP
 * faults. 
 * 
 * @author Scott.Stark@jboss.org
 * @author Rahul Sharma (javadoc)
 * @version $Revision:1$
 */
public class SOAPFaultException extends RuntimeException
{
   /** @since 4.0.2 */
   static final long serialVersionUID;
   private static final int CODE_IDX = 0;
   private static final int STRING_IDX = 1;
   private static final int ACTOR_IDX = 2;
   private static final int DETAIL_IDX = 3;
   private static final ObjectStreamField[] serialPersistentFields;
   static
   {
      if (SerialVersion.version == SerialVersion.LEGACY)
      {
         serialVersionUID = -290987278985292477L;
         serialPersistentFields = new ObjectStreamField[] {
            new ObjectStreamField("faultCode", QName.class),
            new ObjectStreamField("faultString", String.class),
            new ObjectStreamField("faultActor", String.class),
            new ObjectStreamField("faultDetail", Detail.class),
         };
      }
      else
      {
         serialVersionUID = -7224636940495025621L;
         serialPersistentFields = new ObjectStreamField[] {
            new ObjectStreamField("faultcode", QName.class),
            new ObjectStreamField("faultstring", String.class),
            new ObjectStreamField("faultactor", String.class),
            new ObjectStreamField("faultdetail", Detail.class),
         };
      }
   }

   private QName faultCode;
   private String faultString;
   private String faultActor;
   private Detail faultDetail;

   public SOAPFaultException(QName faultCode, String faultString,
      String faultActor, Detail faultDetail)
   {
      this.faultCode = faultCode;
      this.faultString = faultString;
      this.faultActor = faultActor;
      this.faultDetail = faultDetail;
   }

   public String getFaultActor()
   {
      return faultActor;
   }
   public QName getFaultCode()
   {
      return faultCode;
   }
   public Detail getDetail()
   {
      return faultDetail;
   }
   public String getFaultString()
   {
      return faultString;
   }

   // Private -------------------------------------------------------
   private void readObject(ObjectInputStream ois)
      throws ClassNotFoundException, IOException
   {
      ObjectInputStream.GetField fields = ois.readFields();
      String name = serialPersistentFields[CODE_IDX].getName();
      this.faultCode = (QName) fields.get(name, null);
      name = serialPersistentFields[STRING_IDX].getName();
      this.faultString = (String) fields.get(name, null);
      name = serialPersistentFields[ACTOR_IDX].getName();
      this.faultActor = (String) fields.get(name, null);
      name = serialPersistentFields[DETAIL_IDX].getName();
      this.faultDetail = (Detail) fields.get(name, null);
   }

   private void writeObject(ObjectOutputStream oos)
      throws IOException
   {
      // Write j2ee 1.4.1 RI field names
      ObjectOutputStream.PutField fields =  oos.putFields();
      String name = serialPersistentFields[CODE_IDX].getName();
      fields.put(name, faultCode);
      name = serialPersistentFields[STRING_IDX].getName();
      fields.put(name, faultString);
      name = serialPersistentFields[ACTOR_IDX].getName();
      fields.put(name, faultActor);
      name = serialPersistentFields[DETAIL_IDX].getName();
      fields.put(name, faultDetail);
      oos.writeFields();
   }
}
