/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.hibernate.session;

import org.hibernate.Session;
import org.hibernate.HibernateException;
import org.hibernate.SessionFactory;

import javax.naming.InitialContext;
import javax.naming.NamingException;

/**
 * Maintains and exposes, for app usage, the current context bound Hibernate Session.
 * Application code need only deal with the {@link #getSession(java.lang.String)}
 * as the means to retreive the {@link org.hibernate.Session} associated with
 * the current context.
 *
 * @author <a href="mailto:steve@hibernate.org">Steve Ebersole</a>
 * @version $Revision:1$
 *
 * @deprecated Direct use of the new {@link org.hibernate.SessionFactory#getCurrentSession()}
 * method is the preferred approach to managing "transaction contextual sessions".
 */
public class HibernateContext
{
   /**
    * Retreives an "unmanaged" session against the same underlying jdbc connnection as the session
    * currently bound to the current context for the given JNDI name.  This is simply a convenience
    * method for SessionFactory.openSession({@link #getSession}.connection()).  Unmanaged here means that
    * the returned session is not controlled by the code managing the actually bound session; callers
    * are required to cleanup these sessions manually using {@link #releaseUnmanagedSession}.
    *
    * @param name The "name" of the {@link org.hibernate.SessionFactory}
    *       for which an unmanaged session is requested.
    * @return An unmanaged session.
    * @throws HibernateException If an error occurs opening the new Session.
    * @throws IllegalStateException If unable to locate a managed Session for the current context.
    *
    * @deprecated Given a SessionFactory, sf (usually obtained from JNDI), this method is equivalent to
    * <pre>sf.openSession( sf.getCurrentSession().connection() )</pre>
    */
   public static Session getUnmanagedSession(String name) throws HibernateException, IllegalStateException
   {
      SessionFactory sf = locateSessionFactory( name );
      return sf.openSession( sf.getCurrentSession().connection() );
   }

   /**
    * Method to release a previously obtained unmanaged session.
    *
    * @param unmanagedSession The unmanaged Session to release.
    * @throws HibernateException If an error occurs releasing the unmanaged Session.
    *
    * @deprecated See {@link #getUnmanagedSession(String)}
    */
   public static void releaseUnmanagedSession(Session unmanagedSession) throws HibernateException
   {
      unmanagedSession.close();
   }

   /**
    * Retreives the session currently bound to the current context.
    *
    * @param name The "name" of the {@link org.hibernate.SessionFactory}
    *       for which a session is requested.
    * @return The current session.
    *
    * @deprecated Given a SessionFactory, sf (usually obtained from JNDI), this method is equivalent to
    * <pre>sf.getCurrentSession()</pre>.
    * 
    * @see org.hibernate.SessionFactory#getCurrentSession()
    */
   public static Session getSession(String name)
   {
      return locateSessionFactory( name ).getCurrentSession();
   }

   private static SessionFactory locateSessionFactory(String name) throws HibernateException
   {
      InitialContext context = null;
      try
      {
         context = new InitialContext();
         final SessionFactory factory = (SessionFactory) context.lookup(name);
         return factory;
      }
      catch(NamingException e)
      {
         throw new HibernateException("Unable to locate SessionFactory in JNDI under name [" + name + "]", e);
      }
      finally
      {
         release(context);
      }
   }

   private static void release(InitialContext ctx)
   {
      if (ctx != null)
      {
         try
         {
            ctx.close();
         }
         catch(Throwable ignore)
         {
            // ignore
         }
      }
   }
}
