/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.deployment.spi;

// $Id:DeploymentMetaData.java,v 1.0, 2005-06-24 19:01:11Z, Robert Worsnop$

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;
import org.jboss.logging.Logger;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * The deploymant plan is a jar archive that contains deployment meta data and the jboss specific
 * deployment descriptors. The example below shows a top level jboss-app.xml and a jboss.xml
 *
 *    deployment-plan.xml
 *    !/META-INF/jboss-app.xml
 *    testname.jar!/META-INF/jboss.xml
 *
 * @author Thomas.Diesler@jboss.org
 * @since 09-Apr-2004
 */
public class DeploymentMetaData
{
   // provide logging
   private static Logger log = Logger.getLogger(DeploymentMetaData.class);

   /** The entry name in the deployment plan archive */
   public static final String ENTRY_NAME = "deployment-plan.xml";

   private String deploymentName;
   private List entryList = new ArrayList();

   public DeploymentMetaData(String deploymentName)
   {
      this.deploymentName = deploymentName;
   }

   public DeploymentMetaData(Document document)
   {
      init(document);
   }

   public String getDeploymentName()
   {
      return deploymentName;
   }

   public void setDeploymentName(String deploymentName)
   {
      this.deploymentName = deploymentName;
   }

   /**
    * Add an entry and return an id for that entry
    */
   public String addEntry(String archiveName, String descriptorName)
   {
      entryList.add(new Entry(archiveName, descriptorName));

      String entryId = "entry_";
      int count = entryList.size();
      if (count < 100) entryId += "0";
      if (count < 10) entryId += "0";

      return entryId + count;
   }

   public boolean hasEntry(String archiveName, String descriptorName)
   {
      return entryList.contains(new Entry(archiveName, descriptorName));
   }

   public List getEntryList()
   {
      return new ArrayList(entryList);
   }

   public Document getDocument()
   {
      Document document = DocumentHelper.createDocument();
      Element root = document.addElement("jboss-deployment-plan");

      root.addElement("deployment-name").addText(deploymentName);

      root.addComment("Note, deployment-entry elements are not used by the DeploymentManager");
      root.addComment("The DeploymentManager relies on the the entry nameing convention");

      Iterator it = entryList.iterator();
      while (it.hasNext())
      {
         Entry entry = (Entry)it.next();
         Element element = root.addElement("deployment-entry");
         element.addElement("archive-name").addText(entry.archiveName);
         element.addElement("descriptor-name").addText(entry.descriptorName);
      }

      return document;
   }

   public String toXMLString()
   {
      try
      {
         OutputFormat format = OutputFormat.createPrettyPrint();
         StringWriter strWriter = new StringWriter(1024);
         XMLWriter metaWriter = new XMLWriter(strWriter, format);
         metaWriter.write(getDocument());
         metaWriter.close();
         return strWriter.toString();
      }
      catch (IOException e)
      {
         log.error("Cannot get XML string", e);
         return null;
      }
   }

   private void init(Document document)
   {
      Element root = document.getRootElement();
      deploymentName = root.elementTextTrim("deployment-name");
      Iterator it = root.elementIterator("deployment-entry");
      while (it.hasNext())
      {
         Element element = (Element)it.next();
         String archiveName = element.elementTextTrim("archive-name");
         String descriptorName = element.elementTextTrim("descriptor-name");
         addEntry(archiveName, descriptorName);
      }
   }

   /**
    * An entry in the deployment plan
    */
   public static class Entry
   {
      private String archiveName;
      private String descriptorName;

      public Entry(String archiveName, String descriptorName)
      {
         this.archiveName = archiveName;
         this.descriptorName = (descriptorName != null ? descriptorName : "");
      }

      public String getArchiveName()
      {
         return archiveName;
      }

      public String getDescriptorName()
      {
         return descriptorName;
      }

      public boolean equals(Object obj)
      {
         if (obj instanceof Entry)
         {
            Entry other = (Entry)obj;
            return archiveName.equals(other.archiveName) && descriptorName.equals(other.descriptorName);

         }
         return false;
      }

      public int hashCode()
      {
         return new String(archiveName + descriptorName).hashCode();
      }
   }
}
