/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 *
 */
package org.jboss.resource.metadata;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.Locale;

import EDU.oswego.cs.dl.util.concurrent.ConcurrentReaderHashMap;

/**
 * An abstract class for meta data that has descriptions
 *
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @version $Revision:1$
 */
public class DescriptionMetaDataContainer implements Serializable
{
   // Constants -----------------------------------------------------
   static final long serialVersionUID = 2831943526217092377L;
   // Attributes ----------------------------------------------------

   /** The descriptions */
   private ConcurrentReaderHashMap descriptions = new ConcurrentReaderHashMap();
   
   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------

   public DescriptionMetaDataContainer()
   {
      DescriptionMetaData dmd = new DescriptionMetaData();
      descriptions.put(dmd.getLanguage(), dmd);
   }
   
   // Public --------------------------------------------------------

   /**
    * Get the desription for the default language
    * or the first description if there is no default
    * 
    * @return the description for the default language
    */
   public DescriptionMetaData getDescription()
   {
      // Try the default locale
      DescriptionMetaData dmd = (DescriptionMetaData) descriptions.get(Locale.getDefault().getLanguage());
      // No description using the default locale, just use the first
      if (dmd == null)
      {
         for (Iterator i = descriptions.values().iterator(); i.hasNext();)
         {
            dmd = (DescriptionMetaData) i.next();
            break;
         }
      }
      return dmd;
   }
   
   /**
    * Get the description for the give language
    * 
    * @param lang the language
    * @return the description
    */
   public DescriptionMetaData getDescription(String lang)
   {
      return (DescriptionMetaData) descriptions.get(lang);
   }
   
   /**
    * Add a description
    * 
    * @param dmd the description
    */
   public void addDescription(DescriptionMetaData dmd)
   {
      descriptions.put(dmd.getLanguage(), dmd);
   }
   
   /**
    * Get the descriptions
    * 
    * @return the descriptions
    */
   public Collection getDescriptions()
   {
      return descriptions.values();
   }
   
   // Object overrides ----------------------------------------------
   
   public String toString()
   {
      StringBuffer buffer = new StringBuffer();
      buffer.append("DescriptionMetaDataContainer").append('@');
      buffer.append(Integer.toHexString(System.identityHashCode(this)));
      buffer.append("[descriptions=").append(descriptions.values());
      buffer.append(']');
      return buffer.toString();
   }
   
   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------
}
