/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.resource.deployment;

import java.beans.PropertyEditor;
import java.beans.PropertyEditorManager;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Iterator;

import javax.resource.spi.ResourceAdapter;

import org.jboss.deployment.DeploymentException;
import org.jboss.resource.metadata.ConfigPropertyMetaData;
import org.jboss.resource.metadata.ConnectorMetaData;

/**
 * A resource adapter factory
 *
 * @author  <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision:1$
 */
public class ResourceAdapterFactory
{
   // Constants -----------------------------------------------------

   /** The dummy resource adapter for old deployment */
   public static final String DUMMY_RA_CLASS = DummyResourceAdapter.class.getName();
   
   // Attributes ----------------------------------------------------
   
   // Static --------------------------------------------------------

   /**
    * Create a new resource adapter
    * 
    * @param cmd the connector meta data
    * @throws Exception for any error
    */
   public static ResourceAdapter createResourceAdapter(ConnectorMetaData cmd) throws Exception
   {
      // Determine the resource adapter class
      String className = cmd.getRAClass();
      if (className == null)
      {
         if (cmd.getVersion().equals("1.0"))
            className = DUMMY_RA_CLASS;
         else
            throw new IllegalArgumentException("No resource adapter class name specified");
      }
      
      // Load the class
      Class raClass = Thread.currentThread().getContextClassLoader().loadClass(className);
      if (ResourceAdapter.class.isAssignableFrom(raClass) == false)
         throw new DeploymentException(raClass.getName() + " is not a resource adapter class");
      ResourceAdapter result = (ResourceAdapter) raClass.newInstance();
      
      // Apply the properties
      for (Iterator i = cmd.getProperties().iterator(); i.hasNext();)
      {
         ConfigPropertyMetaData cpmd = (ConfigPropertyMetaData) i.next();
         String name = cpmd.getName();
         String type = cpmd.getType();
         String value = cpmd.getValue();
         
         Class clazz = Thread.currentThread().getContextClassLoader().loadClass(type);
         PropertyEditor editor = PropertyEditorManager.findEditor(clazz);
         if (editor == null)
            throw new IllegalArgumentException("No property editor found for property " + cpmd);
         editor.setAsText(value);
         Object object = editor.getValue();
         
         try
         {
            String setter = "set" + name;  
            Method method = raClass.getMethod(setter, new Class[] { clazz });
            method.invoke(result, new Object[] { object });
         }
         catch (InvocationTargetException e)
         {
            DeploymentException.rethrowAsDeploymentException("Error for resource adapter class " + raClass.getName() + " setting property " + cpmd, e.getTargetException());
         }
         catch (Throwable t)
         {
            DeploymentException.rethrowAsDeploymentException("Error for resource adapter class " + raClass.getName() + " accessing property setter " + cpmd, t);
         }
      }
      
      return result;
   }
   
   // Constructors --------------------------------------------------
   
   // Public --------------------------------------------------------
   
   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------
   
   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------

}
