/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.resource.deployment;

import java.util.Properties;

import javax.management.ObjectName;
import javax.naming.InitialContext;

import org.jboss.deployment.DeploymentException;
import org.jboss.naming.Util;
import org.jboss.resource.metadata.AdminObjectMetaData;
import org.jboss.resource.metadata.ConnectorMetaData;
import org.jboss.system.ServiceMBeanSupport;

/**
 * An admin object deployment
 *
 * @author  <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision:1$
 * 
 * @jmx:mbean name="jboss.jca:service=AdminObject"
 *            extends="org.jboss.system.ServiceMBean"
 */
public class AdminObject extends ServiceMBeanSupport implements AdminObjectMBean
{
   // Constants -----------------------------------------------------

   // Attributes ----------------------------------------------------

   /** The resource adapter name */
   protected ObjectName rarName;

   /** The admin object type */
   protected String type;
   
   /** The properties */
   protected Properties properties;
   
   /** The jndi name */
   protected String jndiName;
   
   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------
   
   // Public --------------------------------------------------------
   
   // AdminObjectMBean overrides ------------------------------------

   /**
    * Get the jndi name
    * 
    * @return the jndi name
    * @jmx:managed-attribute
    */
   public String getJNDIName()
   {
      return jndiName;
   }

   /**
    * Set the jndi name
    * 
    * @param jndiName the jndi name
    * @jmx:managed-attribute
    */
   public void setJNDIName(String jndiName)
   {
      this.jndiName = jndiName;
   }

   /**
    * Get the properties
    * 
    * @return the properties
    * @jmx:managed-attribute
    */
   public Properties getProperties()
   {
      return properties;
   }

   /**
    * Set the properties
    * 
    * @param properties the properties
    * @jmx:managed-attribute
    */
   public void setProperties(Properties properties)
   {
      this.properties = properties;
   }

   /**
    * Get the rar name
    * 
    * @return the rar name
    * @jmx:managed-attribute
    */
   public ObjectName getRARName()
   {
      return rarName;
   }

   /**
    * Set the rar name
    * 
    * @param rarName the rar name
    * @jmx:managed-attribute
    */
   public void setRARName(ObjectName rarName)
   {
      this.rarName = rarName;
   }

   /**
    * Get the interface type
    * 
    * @return the interface type
    * @jmx:managed-attribute
    */
   public String getType()
   {
      return type;
   }

   /**
    * Set the interface type
    * 
    * @param type the interface type
    * @jmx:managed-attribute
    */
   public void setType(String type)
   {
      this.type = type;
   }
   
   // ServiceMBeanSupport overrides ---------------------------------
   
   protected void startService() throws Exception
   {
      AdminObjectMetaData aomd = retrieveAdminObjectMetaData();
      if (aomd == null)
         throw new DeploymentException("No admin object metadata type=" + type + " ra=" + rarName);

      Object adminObject = createAdminObject(aomd);
      
      bind(adminObject);
   }
   
   protected void stopService() throws Exception
   {
      unbind();
   }
   
   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   /**
    * Retrieve the admin object metadata
    * 
    * @return the admin object metadata
    * @throws DeploymentException for any error
    */
   protected AdminObjectMetaData retrieveAdminObjectMetaData() throws DeploymentException
   {
      try
      {
         ConnectorMetaData cmd = (ConnectorMetaData) server.getAttribute(rarName, "MetaData");
         return cmd.getAdminObject(type);
      }
      catch (Throwable t)
      {
         DeploymentException.rethrowAsDeploymentException("Error retrieving admin object metadata type=" + type + " ra=" + rarName, t);
         return null; // unreachable
      }
   }

   /**
    * Create the admin object
    * 
    * @param aomd the admin object metadata
    * @return the admin object
    * @throws DeploymentException for any error
    */
   protected Object createAdminObject(AdminObjectMetaData aomd) throws DeploymentException
   {
      try
      {
         return AdminObjectFactory.createAdminObject(jndiName, rarName, aomd, properties);
      }
      catch (Throwable t)
      {
         DeploymentException.rethrowAsDeploymentException("Error creating admin object metadata type=" + type + " ra=" + rarName, t);
         return null; // unreachable
      }
   }

   /**
    * Bind the object into jndi
    * 
    * @param object the object to bind
    * @throws Exception for any error
    */
   protected void bind(Object object) throws Exception
   {
      InitialContext ctx = new InitialContext();
      try
      {
         Util.bind(ctx, jndiName, object);
         log.info("Bound admin object '" + object.getClass().getName() + "' at '" + jndiName + "'");
      }
      finally
      {
         ctx.close();
      }
   }

   /**
    * Unbind the object from jndi
    * 
    * @throws Exception for any error
    */
   protected void unbind() throws Exception
   {
      InitialContext ctx = new InitialContext();
      try
      {
         Util.unbind(ctx, jndiName);
         log.info("Unbound admin object at '" + jndiName + "'");
      }
      finally
      {
         ctx.close();
      }
   }
   
   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------
}
