/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 *
 */
package org.jboss.resource.connectionmanager;

import javax.management.ObjectName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;
import javax.resource.ResourceException;

import org.jboss.deployment.DeploymentException;
import org.jboss.logging.Logger;
import org.jboss.naming.NonSerializableFactory;

import org.jboss.system.ServiceMBeanSupport;

/**
 * Handles the binding of the connection factory into jndi
 *
 * @jmx:mbean name="jboss.jca:service=ConnectionFactoryBinding"
 *            extends="org.jboss.system.ServiceMBean"
 *
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @version $Revision:1$
 */
public class ConnectionFactoryBindingService extends ServiceMBeanSupport 
   implements ConnectionFactoryBindingServiceMBean
{
   // Constants -----------------------------------------------------

   /** The logger */
   private static final Logger log = Logger.getLogger(ConnectionFactoryBindingService.class);
   
   // Attributes ----------------------------------------------------
   
   /** Trace enabled */
   private boolean trace = log.isTraceEnabled();
   
   /** The connection manager */
   protected ObjectName cm;
   
   /** The jndi name */
   protected String jndiName;
   
   /** The bind name */
   protected String bindName;
   
   /** Whether to use the java naming context */
   protected boolean useJavaContext = true;
   
   /** The connection factory */
   protected Object cf;
   
   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------

   // Public --------------------------------------------------------
   
   // ServiceMBeanSupport overrides ---------------------------------
   
   protected void startService() throws Exception
   {
      determineBindName();
      createConnectionFactory();
      bindConnectionFactory();
   }
   
   protected void stopService() throws Exception
   {
      unbindConnectionFactory();
   }
   
   // ConnectionFactoryBindingServiceMBean implementation -----------

   /**
    * Get the connection manager
    * 
    * @jmx:managed-attribute
    * @return the connection manager
    */
   public ObjectName getConnectionManager()
   {
      return cm;
   }

   /**
    * Set the connection manager
    * 
    * @jmx:managed-attribute
    * @param cm the connection manager
    */
   public void setConnectionManager(ObjectName cm)
   {
      this.cm = cm;
   }

   /**
    * Get the bind name
    * 
    * @jmx:managed-attribute
    * @return the real jndi binding
    */
   public String getBindName()
   {
      return bindName;
   }

   /**
    * Get the jndi name
    * 
    * @jmx:managed-attribute
    * @return the jndi name
    */
   public String getJndiName()
   {
      return jndiName;
   }

   /**
    * Set the jndi name
    * 
    * @jmx:managed-attribute
    * @param jndiName the jndi name
    */
   public void setJndiName(String jndiName)
   {
      this.jndiName = jndiName;
   }

   /**
    * Are we using the java naming context
    * 
    * @jmx:managed-attribute
    * @return true when using the java naming context, false otherwise
    */
   public boolean isUseJavaContext()
   {
      return useJavaContext;
   }

   /**
    * Set whether to use the java naming context
    * 
    * @jmx:managed-attribute
    * @param useJavaContext pass true to use the java naming context, false otherwise
    */
   public void setUseJavaContext(boolean useJavaContext)
   {
      this.useJavaContext = useJavaContext;
   }

   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   /**
    * Determine the bind name
    */
   protected void determineBindName() throws Exception
   {
      bindName = jndiName;
      if( useJavaContext && jndiName.startsWith("java:") == false )
         bindName = "java:" + jndiName;
   }

   /**
    * Create the connection factory 
    */
   protected void createConnectionFactory() throws Exception
   {
      try
      {
         BaseConnectionManager2 bcm = (BaseConnectionManager2) server.getAttribute(cm, "Instance");
         BaseConnectionManager2.ConnectionManagerProxy cmProxy = new BaseConnectionManager2.ConnectionManagerProxy(bcm, cm);
         cf = bcm.getPoolingStrategy().getManagedConnectionFactory().createConnectionFactory(cmProxy);
      }
      catch (ResourceException re)
      {
         throw new DeploymentException("Could not create ConnectionFactory for adapter: " + cm);
      }
   }
   
   /**
    * Bind the connection factory into jndi
    */
   protected void bindConnectionFactory() throws Exception
   {
      InitialContext ctx = new InitialContext();
      try
      {
         log.debug("Binding object '" + cf + "' into JNDI at '" + bindName + "'");
         Name name = ctx.getNameParser("").parse(bindName);
         NonSerializableFactory.rebind(name, cf, true);
         log.info("Bound connection factory for resource adapter for ConnectionManager '" + serviceName + " to JNDI name '" + bindName + "'");
      }
      catch (NamingException ne)
      {
         throw new DeploymentException("Could not bind ConnectionFactory into jndi: " + bindName);
      }
      finally
      {
         ctx.close();
      }
   }
   
   /**
    * Unbind the connection factory into jndi
    */
   protected void unbindConnectionFactory() throws Exception
   {
      InitialContext ctx = new InitialContext();
      try
      {
         ctx.unbind(bindName);
         NonSerializableFactory.unbind(bindName);
         log.info("Unbound connection factory for resource adapter for ConnectionManager '" + serviceName + " from JNDI name '" + bindName + "'");
      }
      catch (NamingException ne)
      {
         log.error("Could not unbind managedConnectionFactory from jndi: " + bindName, ne);
      }
      finally
      {
         ctx.close();
      }
   }
   
   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------
}
