/*
 * JBoss, the OpenSource WebOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.resource.adapter.mail.inflow;

import javax.resource.spi.work.Work;
import javax.resource.spi.work.WorkManager;
import javax.resource.spi.work.WorkException;
import javax.resource.spi.work.WorkEvent;
import javax.resource.spi.work.WorkListener;

import EDU.oswego.cs.dl.util.concurrent.BoundedPriorityQueue;
import org.jboss.logging.Logger;

/**
 * @author Scott.Stark@jboss.org
 * @version $Revision:1$
 */
public class NewMsgsWorker
   implements Work, WorkListener
{
   private static Logger log = Logger.getLogger(NewMsgsWorker.class);
   private boolean released;
   private WorkManager mgr;
   private BoundedPriorityQueue pollQueue;
   private boolean trace;

   public NewMsgsWorker(WorkManager mgr)
   {
      this.mgr = mgr;
      // The capacity needs to be externalized
      this.pollQueue = new BoundedPriorityQueue(1024);
      this.trace = log.isTraceEnabled();
   }

   public void watch(MailActivation activation) throws InterruptedException
   {
      long now = System.currentTimeMillis();
      activation.updateNextNewMsgCheckTime(now);
      pollQueue.put(activation);
   }

   public void release()
   {
      released = true;
      if( trace )
         log.trace("released");
   }
   public void run()
   {
      if( trace )
         log.trace("Begin run");
      while( released == false )
      {
         try
         {
            MailActivation ma = (MailActivation) pollQueue.take();
            if( ma.isReleased() )
               continue;
            // Wait until its time to check for new msgs
            long now = System.currentTimeMillis();
            long nextTime = ma.getNextNewMsgCheckTime();
            long sleepMS = nextTime - now;
            Thread.sleep(sleepMS);
            if( released )
               break;

            // Now schedule excecution of the new msg check
            mgr.scheduleWork(ma, WorkManager.INDEFINITE, null, this);
         }
         catch(InterruptedException e)
         {
            log.warn("Interrupted waiting for new msg check", e);
         }
         catch (WorkException e)
         {
            log.warn("Failed to schedule new msg check", e);            
         }
      }
      if( trace )
         log.trace("End run");
   }

   // --- Begin WorkListener interface methods
   public void workAccepted(WorkEvent e)
   {
      if( trace )
         log.trace("workAccepted, e="+e);
   }

   public void workRejected(WorkEvent e)
   {
      if( trace )
         log.trace("workRejected, e="+e);
   }

   public void workStarted(WorkEvent e)
   {
      if( trace )
         log.trace("workStarted, e="+e);
   }

   public void workCompleted(WorkEvent e)
   {
      if( trace )
         log.trace("workCompleted, e="+e);
      MailActivation activation = (MailActivation) e.getWork();
      try
      {
         watch(activation);
      }
      catch(InterruptedException ex)
      {
         log.warn("Failed to reschedule new msg check", ex);
      }
   }
   // --- End WorkListener interface methods

}
