/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.aop;

import java.io.Serializable;

/**
 * Wrapper type for cached AOP instances.
 * When an object is looked up or put in TreeCacheAOP, this object will be advised with a CacheInterceptor.
 * The tree cache stores a reference to this object (for example to update the instance variables, etc.).
 * Since this reference need to be transactional but never replicated (the reference is only valid within the VM)
 * this reference is wrapped into an AOPInstance.
 * In addition, this instance also serves as a metadata for TreeCacheAop. E.g., it has a reference count for
 * multiple references and reference FQN.
 *
 * @author Harald Gliebe
 * @author Ben Wang
 */
public class AOPInstance implements Serializable
{
//    protected static Logger logger_ = Logger.getLogger(AOPInstance.class.getName());
   /**
    * Key under which the AOPInstance is stored in a Node's map.
    */
   // Use the class as key to avoid name clashes
//    public static final Object KEY = AOPInstance.class;
   public static final Object KEY = "AOPInstance";

   // The instance is transient to avoid reflecation outside the VM
   protected transient Object instance_;

   // If not null, it signifies that this is a reference that points to this fqn.
   // Note that this will get replicated.
   protected String refFqn_ = null;

   protected int refCount_ = 0;  // reference counting. THis will get replicated as well.

   public AOPInstance()
   {
   }

   public AOPInstance(Object instance)
   {
      set(instance);
   }

   Object get()
   {
      return instance_;
   }

   void set(Object instance)
   {
      instance_ = instance;
   }

   String getRefFqn()
   {
      return refFqn_;
   }

   void setRefFqn(String refFqn)
   {
      refFqn_ = refFqn;
   }

   void removeRefFqn()
   {
      refFqn_ = null;
   }

   int incrementRefCount()
   {
      refCount_ += 1;
//logger_.info("incrementRefCount(): current ref count " +refCount_);
      return refCount_;
   }

   int decrementRefCount()
   {
      refCount_ -= 1;
//logger_.info("decrementRefCount(): current ref count " +refCount_);
      return refCount_;
   }

   int getRefCount()
   {
      return refCount_;
   }
}
