/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.webservice.transport.jms;

// $Id$

import java.io.InputStream;
import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.Iterator;

import javax.management.MBeanServer;
import javax.xml.namespace.QName;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPMessage;

import org.jboss.axis.AxisFault;
import org.jboss.axis.ConfigurationException;
import org.jboss.axis.EngineConfiguration;
import org.jboss.axis.Message;
import org.jboss.axis.MessageContext;
import org.jboss.axis.configuration.SimpleProvider;
import org.jboss.axis.description.OperationDesc;
import org.jboss.axis.description.ServiceDesc;
import org.jboss.axis.handlers.soap.SOAPService;
import org.jboss.axis.server.AxisServer;
import org.jboss.logging.Logger;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.webservice.AxisServiceMBean;
import org.jboss.webservice.server.InvokerProviderSimple;

/**
 * @author Thomas.Diesler@jboss.org
 */
public class AxisMessageDispatcher implements MessageDispatcher 
{
   // logging support
   protected Logger log = Logger.getLogger(AxisMessageDispatcher.class);

   /* 
    * @see org.jboss.webservice.transport.jms.MessageDispatcher#dipatchMessage(java.lang.String, javax.xml.soap.SOAPMessage)
    */
   public SOAPMessage dipatchMessage(String fromName, Object targetImplBean, InputStream reqMessage) throws RemoteException 
   {
	   String serviceId = "jms-service";
      
      // create the axisRequest and context and invoke the server
      AxisServer axisServer = getDispatchServer(serviceId, targetImplBean);
      MessageContext msgContext = new MessageContext(axisServer);
      
      Message axisResponse = null;
      try
      {
         MessageFactory factory = MessageFactory.newInstance();
         SOAPMessage soapMessage = factory.createMessage(null, reqMessage);
         msgContext.setRequestMessage((Message)soapMessage);
         msgContext.setTargetService(serviceId);
         
         axisServer.invoke(msgContext);
         axisResponse = msgContext.getResponseMessage();
      }
      catch (AxisFault af)
      {
         axisResponse = new Message(af);
         axisResponse.setMessageContext(msgContext);
      }
      catch (Exception e)
      {
         axisResponse = new Message(new AxisFault(e.toString()));
         axisResponse.setMessageContext(msgContext);
      }
      return axisResponse;
   }

   public SOAPMessage delegateMessage(String serviceID, InputStream reqMessage) throws RemoteException
   {
      // create the axisRequest and context and invoke the server
      AxisServer axisServer = getDelegateServer();
      MessageContext msgContext = new MessageContext(axisServer);
      
      Message axisResponse = null;
      try
      {
         MessageFactory factory = MessageFactory.newInstance();
         SOAPMessage soapMessage = factory.createMessage(null, reqMessage);
         msgContext.setRequestMessage((Message)soapMessage);
         
         // maps available service operation QNames to target service names
         // we might want to cache this information
         HashMap serviceOperations = getServiceOperationsMap(axisServer);

         // get the targetService from the SOAPElement
         String targetService = null;
         Iterator it = soapMessage.getSOAPBody().getChildElements();
         while (targetService == null && it.hasNext())
         {
            SOAPElement soapElement = (SOAPElement)it.next();
            String namespace = soapElement.getElementName().getURI();
            String localName = soapElement.getElementName().getLocalName();
            QName qname = new QName(namespace, localName);
            log.debug("maybe operation: " + qname);
            targetService = (String)serviceOperations.get(qname);
         }
         if (targetService != null)
         {
            log.debug("setTargetService: " + targetService);
            msgContext.setTargetService(targetService);
         }
         
         axisServer.invoke(msgContext);
         axisResponse = msgContext.getResponseMessage();
      }
      catch (AxisFault af)
      {
         axisResponse = new Message(af);
         axisResponse.setMessageContext(msgContext);
      }
      catch (Exception e)
      {
         axisResponse = new Message(new AxisFault(e.toString()));
         axisResponse.setMessageContext(msgContext);
      }
      return axisResponse;
   }

   /** Create a new Axis server that dispatches to the given targetBean
    */
   private AxisServer getDispatchServer(String serviceId, Object targetBean)
   {
      AxisServer axisServer;
      try
      {
         SimpleProvider config = new SimpleProvider();
         SOAPService service = new SOAPService(new InvokerProviderSimple(targetBean));
         config.deployService(serviceId, service);
         axisServer = new AxisServer(config);
         log.debug("got AxisServer: " + axisServer);
      }
      catch (Exception e)
      {
         throw new RuntimeException("Cannot obtain axis server", e);
      }
      return axisServer;
   }
   
   /** Get the installed Axis sever.
    */
   protected AxisServer getDelegateServer()
   {
      AxisServer axisServer;
      try
      {
         MBeanServer mbeanServer = MBeanServerLocator.locateJBoss();
         axisServer = (AxisServer)mbeanServer.getAttribute(AxisServiceMBean.OBJECT_NAME, "AxisServer");
         log.debug("got AxisServer: " + axisServer);
      }
      catch (Exception e)
      {
         throw new RuntimeException("Cannot obtain axis server", e);
      }
      return axisServer;
   }
   
   /**
    * Map the available operations for each service to their respective service name
    */
   private HashMap getServiceOperationsMap(AxisServer server)
   {
      HashMap serviceOperations = new HashMap();

      try
      {
         EngineConfiguration config = server.getConfig();
         Iterator it = config.getDeployedServices();
         while (it.hasNext())
         {
            ServiceDesc service = (ServiceDesc)it.next();
            log.debug("service: [name=" + service.getName() + ",ns=" + service.getDefaultNamespace() + "]");
            Iterator opit = service.getOperations().iterator();
            while (opit.hasNext())
            {
               OperationDesc operation = (OperationDesc)opit.next();
               QName qname = operation.getElementQName();
               log.debug("  operation: [qname=" + qname + "]");
               serviceOperations.put(qname, service.getName());
            }
         }
      }
      catch (ConfigurationException e)
      {
         log.error("Cannot map service operations", e);
      }

      return serviceOperations;
   }
}
