/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.webservice.server;

// $Id$

import org.jboss.logging.Logger;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.webservice.AxisServiceMBean;
import org.jboss.webservice.PortComponentInfo;

import javax.management.MBeanServer;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;

/**
 * A servlet that reports the serviceURL for a given service ID.
 * <p/>
 * See {@link org.jboss.webservice.AxisService} for details on the registry of
 * {@link PortComponentInfo} objects.
 * <p/>
 * When the web service client ENC is setup, it may contain port-component-link
 * entries that point to service endpoints in the same top level deployment.
 * The final serviceURL of those endpoints will become available after the
 * reference to the javax.xml.rpc.Service is bound to JNDI.
 * <p/>
 * When the client does a lookup of the javax.xml.rpc.Service from JNDI the ObjectFactory
 * will contact this servlet for the final serviceURL. It is acceptable that the client
 * wsdl does not contain the correct serviceURL if the client is using the port-component-link element.
 *
 * @author Thomas.Diesler@jboss.org
 * @since 29-May-2004
 */
public class PortComponentLinkServlet extends HttpServlet
{
   // provide logging
   private static final Logger log = Logger.getLogger(PortComponentLinkServlet.class);

   /**
    * Get the serviceURL as string for a given serviceID.
    */
   protected void doGet(HttpServletRequest req, HttpServletResponse res) throws ServletException, IOException
   {
      String serviceID = req.getParameter("serviceID");
      if (serviceID == null)
         throw new IllegalArgumentException("Cannot obtain request parameter 'serviceID'");

      PortComponentInfo pcInfo = null;
      try
      {
         MBeanServer server = MBeanServerLocator.locateJBoss();
         pcInfo = (PortComponentInfo)server.invoke(AxisServiceMBean.OBJECT_NAME, "getPortComponentInfo",
                 new Object[]{serviceID}, new String[]{String.class.getName()});
      }
      catch (Exception e)
      {
         throw new ServletException(e);
      }

      if (pcInfo == null)
         throw new IllegalStateException("Cannot resolve port-component-link: " + serviceID);

      res.setContentType("text/plain");
      PrintWriter out = res.getWriter();

      String serviceEndpointURL = pcInfo.getServiceEndpointURL();
      out.println(serviceEndpointURL);

      log.debug("Resolved " + serviceID + " to: " + serviceEndpointURL);

      out.close();
   }
}
