/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.webservice.client;

// $Id$

import java.net.URL;
import java.util.Properties;

import javax.wsdl.Definition;
import javax.xml.namespace.QName;
import javax.xml.rpc.Service;
import javax.xml.rpc.ServiceException;
import javax.xml.rpc.ServiceFactory;

import org.jboss.logging.Logger;
import org.jboss.util.NotImplementedException;
import org.jboss.webservice.deployment.ServiceDescription;
import org.jboss.webservice.metadata.jaxrpcmapping.JavaWsdlMapping;
import org.jboss.webservice.metadata.jaxrpcmapping.JavaWsdlMappingFactory;
import org.jboss.webservice.metadata.wsdl.WSDL11DefinitionFactory;

/**
 * The javax.xml.rpc.ServiceFactory is a factory for the creation of instances of the type javax.xml.rpc.Service.
 * <p/>
 * This class follows the abstract static factory design pattern.
 * This enables a J2SE based client to create a Service instance in a portable
 * manner without using the constructor of the Service implementation class.
 *
 * @author Thomas.Diesler@jboss.org
 * @version 1.1
 */
public class ServiceFactoryImpl extends ServiceFactory
{
   // provide logging
   private final Logger log = Logger.getLogger(ServiceFactoryImpl.class);

   /**
    * Create an instance of the generated service implementation class for a given service interface, if available.
    *
    * @param serviceInterface Service interface
    * @return A Service
    * @throws ServiceException If there is any error while creating the specified service, including the case where a
    *                          generated service implementation class cannot be located
    */
   public Service loadService(Class serviceInterface) throws ServiceException
   {
      throw new NotImplementedException();
   }

   /**
    * Create an instance of the generated service implementation class for a given service interface, if available.
    * An implementation may use the provided wsdlDocumentLocation and properties to help locate the generated implementation class.
    * If no such class is present, a ServiceException will be thrown.
    *
    * @param wsdlDocumentLocation URL for the WSDL document location for the service or null
    * @param serviceInterface     Service interface
    * @param props                A set of implementation-specific properties to help locate the generated service implementation class
    * @return A Service
    * @throws ServiceException If there is any error while creating the specified service, including the case where a
    *                          generated service implementation class cannot be located
    */
   public Service loadService(URL wsdlDocumentLocation, Class serviceInterface, Properties props) throws ServiceException
   {
      throw new NotImplementedException();
   }

   /**
    * Create an instance of the generated service implementation class for a given service, if available.
    * The service is uniquely identified by the wsdlDocumentLocation and serviceName arguments.
    * An implementation may use the provided properties to help locate the generated implementation class.
    * If no such class is present, a ServiceException will be thrown.
    *
    * @param wsdlDocumentLocation URL for the WSDL document location for the service or null
    * @param serviceName          Qualified name for the service
    * @param props                A set of implementation-specific properties to help locate the generated service implementation class
    * @return A Service
    * @throws ServiceException If there is any error while creating the specified service, including the case where a generated service implementation class cannot be located
    */
   public Service loadService(URL wsdlDocumentLocation, QName serviceName, Properties props) throws ServiceException
   {
      throw new NotImplementedException();
   }

   /**
    * Create a <code>Service</code> instance.
    *
    * @param serviceName QName for the service
    * @return Service.
    * @throws ServiceException If any error in creation of the specified service
    */
   public Service createService(QName serviceName) throws ServiceException
   {
      return new ServiceImpl(serviceName);
   }

   /**
    * Create a <code>Service</code> instance.
    *
    * @param wsdlURL URL for the WSDL document location
    * @param serviceName          QName for the service.
    * @return Service.
    * @throws ServiceException If any error in creation of the
    *                          specified service
    */
   public Service createService(URL wsdlURL, QName serviceName) throws ServiceException
   {
      ClassLoader cl = Thread.currentThread().getContextClassLoader();
      
      URL mappingURL = cl.getResource("META-INF/jaxrpc-mapping.xml");
      if (mappingURL != null)
         log.info("Use jaxrpc-mapping from: " + mappingURL);
      
      URL ws4eeURL = cl.getResource("META-INF/ws4ee-deployment.xml");
      if (ws4eeURL != null)
         log.info("Use ws4ee-deployment from: " + ws4eeURL);
      
      return createService(wsdlURL, mappingURL, ws4eeURL, serviceName, null);
   }

   /**
    * Create a <code>Service</code> instance.
    * <p/>
    * Note, this method is not in the {@link ServiceFactory} interface, it provides the service
    * with additional ws4ee wsdl/java mapping information
    *
    * @param wsdlURL    URL for the WSDL document location
    * @param mappingURL An optional URL for the jaxrpc-mapping.xml location
    * @param ws4eeURL An optional URL for the jboss propriatary deployment descriptor, see wiki for details.  
    * @param serviceName     QName for the service.
    * @param portName        An optional port name
    * @return Service.
    * @throws ServiceException If any error in creation of the specified service
    */
   public Service createService(URL wsdlURL, URL mappingURL, URL ws4eeURL, QName serviceName, String portName) throws ServiceException
   {
      ServiceImpl service = new ServiceImpl(wsdlURL, serviceName);

      try
      {
         WSDL11DefinitionFactory wsdlFactory = WSDL11DefinitionFactory.newInstance();
         Definition wsdlDefinition = wsdlFactory.parse(wsdlURL);
         
         if (serviceName != null && portName == null)
         {
            javax.wsdl.Service wsdlService = wsdlDefinition.getService(serviceName);
            if (wsdlService != null && wsdlService.getPorts().size() == 1)
            {
               portName = (String)wsdlService.getPorts().keySet().iterator().next();
            }
         }

         JavaWsdlMapping javaWsdlMapping = null;
         if (mappingURL != null)
         {
            JavaWsdlMappingFactory mappingFactory = JavaWsdlMappingFactory.newInstance();
            javaWsdlMapping = mappingFactory.parse(mappingURL);
         }
         

         ServiceDescription serviceDesc = new ServiceDescription(wsdlDefinition, javaWsdlMapping, ws4eeURL, portName);
         service.initService(serviceDesc, portName);
      }
      catch (Exception e)
      {
         log.error(e.getMessage(), e);
         throw new ServiceException(e);
      }

      return service;
   }
}
