/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.tc5.session;

import org.jboss.cache.Fqn;
import org.jboss.cache.aop.TreeCacheAopMBean;
import org.jboss.cache.lock.TimeoutException;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class JBossCacheWrapper {
   private static final int RETRY = 3;
   private TreeCacheAopMBean proxy_;

   JBossCacheWrapper(TreeCacheAopMBean cache)
   {
      proxy_ = cache;
   }

   /**
    * Wrapper to embed retyr logic.
    *
    * @param fqn
    * @param id
    * @return
    */
   Object get(Fqn fqn, String id)
   {
      Exception ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            return proxy_.get(fqn, id);
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("JBossCacheService: exception occurred in cache get ... ", e);
         }
      }
      throw new RuntimeException("JBossCacheService: exception occurred in cache get after retry ... ", ex);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    * @param id
    * @param value
    * @return
    */
   Object put(Fqn fqn, String id, Object value)
   {
      Exception ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            return proxy_.put(fqn, id, value);
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("JBossCacheService: exception occurred in cache put ... ", e);
         }
      }
      throw new RuntimeException("JBossCacheService: exception occurred in cache put after retry ... ", ex);
   }


   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    * @param map
    */
   void put(Fqn fqn, Map map)
   {
      Exception ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            proxy_.put(fqn, map);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("JBossCacheService: exception occurred in cache put ... ", e);
         }
      }
      throw new RuntimeException("JBossCacheService: exception occurred in cache put after retry ... ", ex);
   }

   /**
    * Wrapper to embed retyr logic.
    *
    * @param fqn
    * @param id
    * @return
    */
   Object remove(Fqn fqn, String id)
   {
      Exception ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            return proxy_.remove(fqn, id);
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("JBossCacheService: exception occurred in cache remove ... ", e);
         }
      }
      throw new RuntimeException("JBossCacheService: exception occurred in cache remove after retry ... ", ex);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    */
   void remove(Fqn fqn)
   {
      Exception ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            proxy_.remove(fqn);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("JBossCacheService: exception occurred in cache remove ... ", e);
         }
      }
      throw new RuntimeException("JBossCacheService: exception occurred in cache remove after retry ... ", ex);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    */
   void evict(Fqn fqn)
   {
      Exception ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            proxy_.evict(fqn);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("JBossCacheService: exception occurred in cache evict ... ", e);
         }
      }
      throw new RuntimeException("JBossCacheService: exception occurred in cache evict after retry ... ", ex);
   }
   
   void evictSubtree(Fqn fqn)
   {
      // First evict the node itself.  If it is the root of an AOP
      // object tree, this will do everything
      evict(fqn);

      // Next, clear any children
      
      Set children = null;
      Exception ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            children = proxy_.getChildrenNames(fqn);
            ex = null;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("JBossCacheService: exception occurred in cache evict ... ", e);
         }
      }
      
      if (ex != null)
         throw new RuntimeException("JBossCacheService: exception occurred in cache evictSubtree after retry ... ", ex);
      
      // Evict any children
      if (children != null)
      {
         for (Iterator it = children.iterator(); it.hasNext(); )
         {
            Fqn child = new Fqn(fqn, it.next());
            evict(child);
         }
      }
   }

}
