/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.test.web.test;

import java.net.HttpURLConnection;

import junit.framework.Test;

import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpState;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;

import org.jboss.jmx.adaptor.rmi.RMIAdaptor;
import org.jboss.test.JBossClusteredTestCase;


/** Tests of web app single sign-on in a clustered environment
 * 
 * TODO general refactoring; a lot of duplicated code here
 * 
 * @author Brian Stansberry
 * @version $Revision$
 */
public class ClusteredSingleSignOnUnitTestCase 
      extends JBossClusteredTestCase
{   
   private boolean deployed0 = true;
   private boolean deployed1 = true;
   private RMIAdaptor[] adaptors = null;
   
   public ClusteredSingleSignOnUnitTestCase(String name)
   {
      super(name);
   }

   /** One time setup for all ClusteredSingleSignOnUnitTestCase unit tests
    */
   public static Test suite() throws Exception
   {
      Test t1 = JBossClusteredTestCase.getDeploySetup(ClusteredSingleSignOnUnitTestCase.class,
            "web-sso-clustered.ear");
      return t1;
   }

   protected void setUp() throws Exception
   {
      super.setUp();
      
      adaptors = getAdaptors(); 
      if (!deployed0)
      {
         deploy(adaptors[0], "web-sso-clustered.ear");
         deployed0 = true;
      }
      if (!deployed1)
      {
         deploy(adaptors[1], "web-sso-clustered.ear");
         deployed1 = true;
      }
   }

   /** Test single sign-on across two web apps using form based auth
    * 
    * @throws Exception
    */ 
   public void testFormAuthSingleSignOn() throws Exception
   {
      log.info("+++ testFormAuthSingleSignOn");
      String[] httpURLs  = super.getHttpURLs();

      String serverA = httpURLs[0];
      String serverB = httpURLs[1];
      log.info(System.getProperties());
      log.info("serverA: "+serverA);
      log.info("serverB: "+serverB);
      SSOBaseCase.executeFormAuthSingleSignOnTest(serverA, serverB, getLog());
   }
   
   /** 
    * Tests that use of transactions in ClusteredSSO does not interfere 
    * with session expiration thread.  See JBAS-2212.
    * 
    * @throws Exception
    */ 
   public void testSessionExpiration() 
         throws Exception
   {
      log.info("+++ testSessionExpiration");
      String[] httpURLs  = super.getHttpURLs();

      String serverA = httpURLs[0];
      log.info(System.getProperties());
      log.info("serverA: "+serverA);
      
      // Start by accessing the secured index.html of war1
      HttpClient httpConn = new HttpClient();
      GetMethod indexGet = new GetMethod(serverA+"/war3/index.jsp");
      int responseCode = httpConn.executeMethod(indexGet);
      String body = indexGet.getResponseBodyAsString();
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      assertTrue("Redirected to login page", body.indexOf("j_security_check") > 0 );

      HttpState state = httpConn.getState();
      Cookie[] cookies = state.getCookies();
      String sessionID = null;
      for(int c = 0; c < cookies.length; c ++)
      {
         Cookie k = cookies[c];
         if( k.getName().equalsIgnoreCase("JSESSIONID") )
            sessionID = k.getValue();
      }
      log.debug("Saw JSESSIONID="+sessionID);

      // Submit the login form
      PostMethod formPost = new PostMethod(serverA+"/war3/j_security_check");
      formPost.addRequestHeader("Referer", serverA+"/war3/login.html");
      formPost.addParameter("j_username", "jduke");
      formPost.addParameter("j_password", "theduke");
      responseCode = httpConn.executeMethod(formPost.getHostConfiguration(),
         formPost, state);
      String response = formPost.getStatusText();
      log.debug("responseCode="+responseCode+", response="+response);
      assertTrue("Saw HTTP_MOVED_TEMP("+responseCode+")",
         responseCode == HttpURLConnection.HTTP_MOVED_TEMP);

      //  Follow the redirect to the index.html page
      Header location = formPost.getResponseHeader("Location");
      String indexURI = location.getValue();
      GetMethod war1Index = new GetMethod(indexURI);
      responseCode = httpConn.executeMethod(war1Index.getHostConfiguration(),
         war1Index, state);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      body = war1Index.getResponseBodyAsString();
      if( body.indexOf("j_security_check") > 0 )
         fail("get of "+indexURI+" redirected to login page");

      // Wait more than 65 secs to let session time out
      // 5 secs for the session timeout, 2 * 30 secs for the processor thread
      // TODO for some reason it takes 1 min for processExpires to run ???
      // (not an sso issue -- a tomcat issue)
      try {
         Thread.sleep(65500);
      }
      catch (InterruptedException ie)
      {
         log.debug("Interrupted while waiting for session expiration");
      }
      
      // Try accessing war1 again
      war1Index = new GetMethod(serverA+"/war3/index.jsp");
      responseCode = httpConn.executeMethod(war1Index.getHostConfiguration(),
         war1Index, state);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      body = war1Index.getResponseBodyAsString();
      log.debug("body: " + body);
      if( body.indexOf("j_security_check") < 0 )
         fail("get of /war1/index.html not redirected to login page");
      
   }
   
   /**
    * Tests that undeploying a webapp on one server doesn't kill an sso
    * that also has a session from another webapp associated with it.
    * See JBAS-2429.
    * 
    * TODO create an independently deployable war so we can test this in
    *      a non-clustered environment as well; this isn't a clustering issue 
    *
    * @throws Exception
    */
   public void testWebappUndeploy() throws Exception
   {
      log.info("+++ testWebappUndeploy");
      
      String[] httpURLs  = super.getHttpURLs();

      String serverA = httpURLs[0];
      String serverB = httpURLs[1];
      
      // Start by accessing the secured index.html of war1
      HttpClient httpConn = new HttpClient();
      GetMethod indexGet = new GetMethod(serverA+"/war1/index.html");
      int responseCode = httpConn.executeMethod(indexGet);
      String body = indexGet.getResponseBodyAsString();
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      assertTrue("Redirected to login page", body.indexOf("j_security_check") > 0 );

      HttpState state = httpConn.getState();
      Cookie[] cookies = state.getCookies();
      String sessionID = null;
      for(int c = 0; c < cookies.length; c ++)
      {
         Cookie k = cookies[c];
         if( k.getName().equalsIgnoreCase("JSESSIONID") )
            sessionID = k.getValue();
}
      log.debug("Saw JSESSIONID="+sessionID);
      // Submit the login form
      PostMethod formPost = new PostMethod(serverA+"/war1/j_security_check");
      formPost.addRequestHeader("Referer", serverA+"/war1/login.html");
      formPost.addParameter("j_username", "jduke");
      formPost.addParameter("j_password", "theduke");
      responseCode = httpConn.executeMethod(formPost.getHostConfiguration(),
         formPost, state);
      String response = formPost.getStatusText();
      log.debug("responseCode="+responseCode+", response="+response);
      assertTrue("Saw HTTP_MOVED_TEMP("+responseCode+")",
         responseCode == HttpURLConnection.HTTP_MOVED_TEMP);

      //  Follow the redirect to the index.html page
      Header location = formPost.getResponseHeader("Location");
      String indexURI = location.getValue();
      GetMethod war1Index = new GetMethod(indexURI);
      responseCode = httpConn.executeMethod(war1Index.getHostConfiguration(),
         war1Index, state);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      body = war1Index.getResponseBodyAsString();
      if( body.indexOf("j_security_check") > 0 )
         fail("get of "+indexURI+" redirected to login page");

      cookies = state.getCookies();
      String ssoID = null;
      for(int c = 0; c < cookies.length; c ++)
      {
         Cookie k = cookies[c];
         if( k.getName().equalsIgnoreCase("JSESSIONIDSSO") )
         {
            ssoID = k.getValue();
            // Make an sso cookie to send to serverB
            Cookie copy = SSOBaseCase.copyCookie(k, serverB);
            state.addCookie(copy);
            log.debug("Added state cookie: "+copy);
         }
      }
      assertTrue("Saw JSESSIONIDSSO", ssoID != null);
      log.debug("Saw JSESSIONIDSSO="+ssoID);

      // Pause a moment before switching wars to better simulate real life
      // use cases.  Otherwise, the test case can "outrun" the async
      // replication in the TreeCache used by the clustered SSO
      // 500 ms is a long time, but this isn't a test of replication speed
      // and we don't want spurious failures.
      if (!serverA.equals(serverB))
         Thread.sleep(500);

      // Now try getting the war2 index using the JSESSIONIDSSO cookie 
      log.debug("Prepare /war2/index.html get");
      GetMethod war2Index = new GetMethod(serverB+"/war2/index.html");
      responseCode = httpConn.executeMethod(war2Index.getHostConfiguration(),
         war2Index, state);
      response = war2Index.getStatusText();
      log.debug("responseCode="+responseCode+", response="+response);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      body = war2Index.getResponseBodyAsString();
      log.debug("body: "+body);
      if( body.indexOf("j_security_check") > 0 )
         fail("get of /war2/index.html redirected to login page");

      // Sleep some more to allow the updated sso to propagate back to serverA
      if (!serverA.equals(serverB))
         Thread.sleep(500);
      
      // We now have a clustered sso context, plus a war1 session on
      // serverA and a war2 session on serverB. No war1 session on serverB,
      // so the only way to access war1 on B without a login is through sso.
      
      //Undeploy the ear from serverA and confirm that it doesn't kill the sso
      undeploy(adaptors[0], "web-sso-clustered.ear");
      deployed0 = false;

      // Sleep some more to allow the updated sso to propagate back to serverB
      if (!serverA.equals(serverB))
         Thread.sleep(500);
      
      //    Now try getting the war1 index using the JSESSIONIDSSO cookie 
      log.debug("Prepare /war1/index.html get");
      war1Index = new GetMethod(serverB+"/war1/index.html");
      responseCode = httpConn.executeMethod(war1Index.getHostConfiguration(),
         war1Index, state);
      response = war1Index.getStatusText();
      log.debug("responseCode="+responseCode+", response="+response);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      body = war1Index.getResponseBodyAsString();
      log.debug("body: "+body);
      if( body.indexOf("j_security_check") > 0 )
         fail("get of /war1/index.html redirected to login page");
   }
}
