/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.spring.factory;

import org.springframework.beans.factory.xml.DefaultXmlBeanDefinitionParser;
import org.springframework.beans.factory.BeanDefinitionStoreException;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.Node;
import org.jboss.logging.Logger;

import java.util.regex.Pattern;
import java.util.regex.Matcher;

/**
 * @author <a href="mailto:ales.justin@genera-lynx.com">Ales Justin</a>
 */
public class NamedXmlBeanDefinitionParser extends DefaultXmlBeanDefinitionParser implements Nameable {

    private static final Logger log = Logger.getLogger(NamedXmlBeanDefinitionParser.class);
    public static final String BEAN_FACTORY_ELEMENT = "BeanFactory=\\(([^)]+)\\)";
    public static final String PARENT_BEAN_FACTORY_ELEMENT = "ParentBeanFactory=\\(([^)]+)\\)";

    private ConfigurableBeanFactory beanFactory;
    private String beanFactoryName;

    public NamedXmlBeanDefinitionParser(ConfigurableBeanFactory beanFactory) {
        this.beanFactory = beanFactory;
    }

    protected void preProcessXml(Element root) throws BeanDefinitionStoreException {
        NodeList nl = root.getChildNodes();
        for (int i = 0; i < nl.getLength(); i++) {
            Node node = nl.item(i);
            if (node instanceof Element) {
                Element ele = (Element) node;
                if (DESCRIPTION_ELEMENT.equals(node.getNodeName())) {
                    String nodeValue = ele.getFirstChild().getNodeValue();
                    log.info("Bean names [description tag]: " + nodeValue);
                    Matcher bfm = parse(nodeValue, BEAN_FACTORY_ELEMENT);
                    if (bfm.find()) {
                        beanFactoryName = bfm.group(1);
                    }
                    Matcher pbfm = parse(nodeValue, PARENT_BEAN_FACTORY_ELEMENT);
                    if (pbfm.find()) {
                        String parentBeanFactoryName = pbfm.group(1);
                        try {
                            beanFactory.setParentBeanFactory(BeanFactoryLoaderImpl.lookup(parentBeanFactoryName));
                        } catch (Exception e) {
                            throw new BeanDefinitionStoreException("Failure during parent bean factory JNDI lookup: " + parentBeanFactoryName, e);
                        }
                    }
                }
            }
        }
    }

    public String getName() {
        return beanFactoryName;
    }

    private static Matcher parse(String value, String regexp) {
        Pattern pattern = Pattern.compile(regexp);
        return pattern.matcher(value);
    }

}
