/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.spring.deployment;

import org.jboss.deployment.DeploymentException;
import org.jboss.deployment.DeploymentInfo;
import org.jboss.deployment.SubDeployer;
import org.jboss.deployment.SubDeployerSupport;
import org.jboss.spring.factory.BeanFactoryLoader;
import org.jboss.spring.factory.BeanFactoryLoaderImpl;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.ObjectName;
import java.io.File;
import java.net.URL;

/**
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author <a href="mailto:ales.justin@genera-lynx.com">Ales Justin</a>
 * @jmx:mbean name="jboss.spring:service=SpringDeployer"
 *            extends="org.jboss.deployment.SubDeployerMBean"
 */
public class SpringDeployer extends SubDeployerSupport implements
        SubDeployer, SpringDeployerMBean {

    private BeanFactoryLoader beanFactoryLoader = new BeanFactoryLoaderImpl();

    /**
     * Default CTOR used to set default values to the Suffixes and RelativeOrder
     * attributes. Those are read at subdeployer registration time by the MainDeployer
     * to alter its SuffixOrder.
     */
    public SpringDeployer() {
        initializeMainDeployer();
    }

    protected void initializeMainDeployer() {
        setSuffixes(new String[]{".spring", "-spring.xml"});
        setRelativeOrder(RELATIVE_ORDER_200);
    }

    /**
     * Returns true if this deployer can deploy the given DeploymentInfo.
     *
     * @return True if this deployer can deploy the given DeploymentInfo.
     * @jmx:managed-operation
     */
    public boolean accepts(DeploymentInfo di) {
        String urlStr = di.url.toString();
        return urlStr.endsWith(".spring") || urlStr.endsWith(".spring/") ||
                urlStr.endsWith("-spring.xml");
    }

    /**
     * Describe <code>init</code> method here.
     *
     * @param di a <code>DeploymentInfo</code> value
     * @throws DeploymentException if an error occurs
     * @jmx:managed-operation
     */
    public void init(DeploymentInfo di) throws DeploymentException {
        try {
            if (di.watch == null) {
                // resolve the watch
                if (di.url.getProtocol().equals("file")) {
                    File file = new File(di.url.getFile());
                    // If not directory we watch the package
                    if (!file.isDirectory()) {
                        di.watch = di.url;
                    }
                    // If directory we watch the xml files
                    else {
                        di.watch = new URL(di.url, "META-INF/jboss-spring.xml");
                    }
                } else {
                    // We watch the top only, no directory support
                    di.watch = di.url;
                }
            }
        } catch (Exception e) {
            log.error("failed to parse Spring context document: ", e);
            throw new DeploymentException(e);
        }
        super.init(di);
    }

    /**
     * Describe <code>create</code> method here.
     *
     * @param di a <code>DeploymentInfo</code> value
     * @throws DeploymentException if an error occurs
     * @jmx:managed-operation
     */
    public void create(DeploymentInfo di) throws DeploymentException {
        try {
            beanFactoryLoader.create(di);
            Notification msg = new Notification("Spring Deploy", this,
                                                getNextNotificationSequenceNumber());
            sendNotification(msg);
            log.info("Deployed Spring: " + di.url);
        } catch (Exception e) {
            throw new DeploymentException(e);
        }
    }

    /**
     * The <code>start</code> method starts all the mbeans in this DeploymentInfo..
     *
     * @param di a <code>DeploymentInfo</code> value
     * @throws DeploymentException if an error occurs
     * @jmx:managed-operation
     */
    public void start(DeploymentInfo di) throws DeploymentException {
        beanFactoryLoader.start(di);
    }

    /**
     * Undeploys the package at the url string specified. This will: Undeploy
     * packages depending on this one. Stop, destroy, and unregister all the
     * specified mbeans Unload this package and packages this package deployed
     * via the classpath tag. Keep track of packages depending on this one that
     * we undeployed so that they can be redeployed should this one be
     * redeployed.
     *
     * @param di the <code>DeploymentInfo</code> value to stop.
     * @jmx:managed-operation
     */
    public void stop(DeploymentInfo di) {
        log.info("Undeploying Spring: " + di.url);
        try {
            beanFactoryLoader.stop(di);
            Notification msg = new Notification("Spring Undeploy", this,
                                                getNextNotificationSequenceNumber());
            sendNotification(msg);
        } catch (Exception e) {
            log.error("Failed to stop bean factory: " + di.url);
        }
    }

    /**
     * Describe <code>destroy</code> method here.
     *
     * @param di a <code>DeploymentInfo</code> value
     * @jmx:managed-operation
     */
    public void destroy(DeploymentInfo di) {
        try {
            beanFactoryLoader.destroy(di);
        } catch (DeploymentException e) {
            log.error("Failed to destroy deployer: " + di);
        }
    }

    protected ObjectName getObjectName(MBeanServer server, ObjectName name)
            throws MalformedObjectNameException {
        return name == null ? OBJECT_NAME : name;
    }

}
