/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.ejb.plugins.cmp.jdbc2;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import javax.ejb.FinderException;
import org.jboss.ejb.GenericEntityObjectFactory;
import org.jboss.ejb.plugins.cmp.jdbc.EJBQLToSQL92Compiler;
import org.jboss.ejb.plugins.cmp.jdbc.QLCompiler;
import org.jboss.ejb.plugins.cmp.jdbc.QueryParameter;
import org.jboss.ejb.plugins.cmp.jdbc.metadata.JDBCDynamicQLQueryMetaData;
import org.jboss.ejb.plugins.cmp.jdbc2.bridge.JDBCCMPFieldBridge2;
import org.jboss.ejb.plugins.cmp.jdbc2.bridge.JDBCEntityBridge2;
import org.jboss.ejb.plugins.cmp.jdbc2.schema.Schema;
import org.jboss.logging.Logger;

/**
 * @author <a href="mailto:alex@jboss.org">Alexey Loubyansky</a>
 * @version <tt>$Revision$</tt>
 */
public class DynamicQueryCommand
   implements QueryCommand
{
   private Logger log;
   private JDBCEntityBridge2 entity;
   private JDBCDynamicQLQueryMetaData metadata;
   private AbstractQueryCommand.CollectionFactory collectionFactory;

   public DynamicQueryCommand(JDBCEntityBridge2 entity, JDBCDynamicQLQueryMetaData metadata)
   {
      log =
         Logger.getLogger(getClass().getName() + "." + entity.getEntityName() + "#" + metadata.getMethod().getName());
      this.entity = entity;
      this.metadata = metadata;

      Class returnType = metadata.getMethod().getReturnType();
      if(Collection.class.isAssignableFrom(returnType))
      {
         if(Set.class.isAssignableFrom(returnType))
         {
            collectionFactory = AbstractQueryCommand.SET_FACTORY;
         }
         else
         {
            collectionFactory = AbstractQueryCommand.COLLECTION_FACTORY;
         }
      }
   }

   public JDBCStoreManager2 getStoreManager()
   {
      return (JDBCStoreManager2)entity.getManager();
   }

   public Collection fetchCollection(Schema schema, GenericEntityObjectFactory factory, Object[] args)
      throws FinderException
   {
      if(log.isTraceEnabled())
      {
         log.trace("executing dynamic-ql: " + args[0]);
      }

      JDBCStoreManager2 manager = (JDBCStoreManager2)entity.getManager();
      QLCompiler compiler = new EJBQLToSQL92Compiler(manager.getCatalog());
      try
      {
         compiler.compileJBossQL((String)args[0],
            metadata.getMethod().getReturnType(),
            getParamTypes(args),
            metadata
         );
      }
      catch(Throwable t)
      {
         log.error("Error compiling JBossQL statement '" + args[0] + "'", t);
         throw new FinderException("Error compiling JBossQL statement '" + args[0] + "'");
      }

      String sql = compiler.getSQL();

      AbstractQueryCommand.ResultReader resultReader;
      if(!compiler.isSelectEntity())
      {
         if(compiler.isSelectField())
         {
            resultReader = new AbstractQueryCommand.FieldReader((JDBCCMPFieldBridge2)compiler.getSelectField());
         }
         else
         {
            resultReader = new AbstractQueryCommand.FunctionReader(compiler.getSelectFunction());
         }
      }
      else
      {
         resultReader = new AbstractQueryCommand.EntityReader((JDBCEntityBridge2)compiler.getSelectEntity());
      }

      return AbstractQueryCommand.fetchCollection(
         entity, sql, toArray(compiler.getInputParameters()),
         new AbstractQueryCommand.EagerCollectionStrategy(collectionFactory, resultReader, log),
         schema, factory, (Object[])args[1], log);
   }

   public Object fetchOne(Schema schema, GenericEntityObjectFactory factory, Object[] args) throws FinderException
   {
      if(log.isTraceEnabled())
      {
         log.trace("executing dynamic-ql: " + args[0]);
      }

      JDBCStoreManager2 manager = (JDBCStoreManager2)entity.getManager();
      QLCompiler compiler = new EJBQLToSQL92Compiler(manager.getCatalog());
      try
      {
         compiler.compileJBossQL((String)args[0],
            metadata.getMethod().getReturnType(),
            getParamTypes(args),
            metadata
         );
      }
      catch(Throwable t)
      {
         log.error("Error compiling JBossQL statement '" + args[0] + "'", t);
         throw new FinderException("Error compiling JBossQL statement '" + args[0] + "'");
      }

      String sql = compiler.getSQL();

      AbstractQueryCommand.ResultReader resultReader;
      if(!compiler.isSelectEntity())
      {
         if(compiler.isSelectField())
         {
            resultReader = new AbstractQueryCommand.FieldReader((JDBCCMPFieldBridge2)compiler.getSelectField());
         }
         else
         {
            resultReader = new AbstractQueryCommand.FunctionReader(compiler.getSelectFunction());
         }
      }
      else
      {
         resultReader = new AbstractQueryCommand.EntityReader((JDBCEntityBridge2)compiler.getSelectEntity());
      }

      return AbstractQueryCommand.fetchOne(entity, sql, toArray(compiler.getInputParameters()),
         resultReader, (Object[])args[1], factory, log
      );
   }

   private static Class[] getParamTypes(Object[] args)
      throws FinderException
   {
      Class[] parameterTypes;
      // get the parameters
      Object[] parameters = (Object[])args[1];
      if(parameters == null)
      {
         parameterTypes = new Class[0];
      }
      else
      {
         // get the parameter types
         parameterTypes = new Class[parameters.length];
         for(int i = 0; i < parameters.length; i++)
         {
            if(parameters[i] == null)
            {
               throw new FinderException("Parameter[" + i + "] is null");
            }
            parameterTypes[i] = parameters[i].getClass();
         }
      }
      return parameterTypes;
   }

   static QueryParameter[] toArray(List p)
   {
      QueryParameter[] params = null;
      if(p.size() > 0)
      {
         params = (QueryParameter[])p.toArray(new QueryParameter[p.size()]);
      }
      return params;
   }
}
