/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2002-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.jboss.axis.soap;

import org.jboss.axis.AxisFault;
import org.jboss.axis.Constants;
import org.jboss.axis.Message;
import org.jboss.axis.attachments.Attachments;
import org.jboss.axis.client.Call;
import org.jboss.axis.message.SOAPEnvelopeAxisImpl;
import org.jboss.axis.transport.http.HTTPConstants;
import org.jboss.axis.utils.Messages;
import org.jboss.logging.Logger;
import org.w3c.dom.Element;

import javax.xml.soap.MimeHeader;
import javax.xml.soap.MimeHeaders;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;
import java.net.MalformedURLException;
import java.util.Hashtable;
import java.util.Iterator;

/**
 * SOAP Connection implementation
 *
 * @author Davanum Srinivas (dims@yahoo.com)
 */
public class SOAPConnectionImpl extends javax.xml.soap.SOAPConnection
{
   boolean closed = false;

   private static Logger log = Logger.getLogger(SOAPConnectionImpl.class.getName());

   /**
    * Sends the given message to the specified endpoint and
    * blocks until it has returned the response.
    *
    * @param request  the <CODE>SOAPMessage</CODE>
    *                 object to be sent
    * @param endpoint a <CODE>URLEndpoint</CODE>
    *                 object giving the URL to which the message should be
    *                 sent
    * @return the <CODE>SOAPMessage</CODE> object that is the
    *         response to the message that was sent
    * @throws SOAPException if there is a SOAP error
    */
   public SOAPMessage call(SOAPMessage request, Object endpoint)
           throws SOAPException
   {
      if (closed)
      {
         throw new SOAPException(Messages.getMessage("connectionClosed00"));
      }
      try
      {
         Call call = createCall(endpoint);
         ((org.jboss.axis.Message)request).setMessageContext(call.getMessageContext());
         SOAPEnvelopeAxisImpl env = ((org.jboss.axis.Message)request).getSOAPEnvelope();
         Attachments attachments = ((org.jboss.axis.Message)
                 request).getAttachmentsImpl();
         if (attachments != null)
         {
            Iterator iterator = attachments.getAttachments().iterator();
            while (iterator.hasNext())
            {
               Object attachment = iterator.next();
               call.addAttachmentPart(attachment);
            }
         }

         call.setReturnClass(SOAPMessage.class);
         call.invoke(env, request.getMimeHeaders());
         return call.getResponseMessage();
      }
      catch (java.net.MalformedURLException mue)
      {
         throw new SOAPException(mue);
      }
      catch (org.jboss.axis.AxisFault af)
      {
         processAxisFault(af);
         Message msg = new Message(af);
         msg.setProperty(SOAPMessage.WRITE_XML_DECLARATION, "true");
         msg.setMessageContext(((org.jboss.axis.Message)request).getMessageContext());
         return msg;
//            throw new SOAPException(af);
      }
      catch (java.rmi.RemoteException re)
      {
         throw new SOAPException(re);
      }
   }

   /**
    * routine called whenever an axis fault is caught; where they
    * are logged and any other business. The method may modify the fault
    * in the process
    *
    * @param fault what went wrong.
    */
   protected void processAxisFault(AxisFault fault)
   {
      //log the fault
      Element runtimeException = fault.lookupFaultDetail(Constants.QNAME_FAULTDETAIL_STACKTRACE);
      if (runtimeException != null)
      {
         //strip runtime details
         fault.removeFaultDetail(Constants.QNAME_FAULTDETAIL_STACKTRACE);
      }
   }

   /**
    * Stubbed out to allow subclassing
    */
   protected Call createCall(Object endpoint) throws MalformedURLException
   {
      Call call = new Call(endpoint.toString());
      return call;
   }

   /**
    * Closes this <CODE>SOAPConnection</CODE> object.
    *
    * @throws SOAPException if there is a SOAP error
    */
   public void close() throws SOAPException
   {
      if (closed)
      {
         throw new SOAPException(Messages.getMessage("connectionClosed00"));
      }
      closed = true;
   }
}
