/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.jboss.axis.handlers.soap;

import org.jboss.axis.AxisEngine;
import org.jboss.axis.AxisFault;
import org.jboss.axis.Constants;
import org.jboss.axis.Handler;
import org.jboss.axis.MessageContext;
import org.jboss.axis.SimpleTargetedChain;
import org.jboss.axis.attachments.Attachments;
import org.jboss.axis.description.ServiceDesc;
import org.jboss.axis.encoding.TypeMappingRegistry;
import org.jboss.axis.enums.Style;
import org.jboss.axis.enums.Use;
import org.jboss.axis.handlers.BasicHandler;
import org.jboss.axis.handlers.HandlerChainImpl;
import org.jboss.axis.handlers.HandlerInfoChainFactory;
import org.jboss.axis.message.SOAPEnvelopeAxisImpl;
import org.jboss.axis.message.SOAPHeaderElementAxisImpl;
import org.jboss.axis.providers.BasicProvider;
import org.jboss.axis.soap.SOAPConstants;
import org.jboss.axis.utils.LockableHashtable;
import org.jboss.axis.utils.Messages;
import org.jboss.axis.utils.XMLUtils;
import org.jboss.logging.Logger;
import org.w3c.dom.Document;

import javax.xml.namespace.QName;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;


/**
 * A <code>SOAPService</code> is a Handler which encapsulates a SOAP
 * invocation.  It has an request chain, an response chain, and a pivot-point,
 * and handles the SOAP semantics when invoke()d.
 *
 * @author Glen Daniels (gdaniels@macromedia.com)
 * @author Doug Davis (dug@us.ibm.com)
 */
public class SOAPService extends SimpleTargetedChain
{
   private static Logger log = Logger.getLogger(SOAPService.class.getName());

   /**
    * Valid transports for this service
    * (server side only!)
    * <p/>
    * !!! For now, if this is null, we assume all
    * transports are valid.
    */
   private Vector validTransports = null;

   /**
    * Does this service require a high-fidelity SAX recording of messages?
    * (default is true)
    */
   private boolean highFidelityRecording = true;

   /**
    * How does this service wish data which would normally be sent as
    * an attachment to be sent?  Default for requests is
    * org.jboss.axis.attachments.Attachments.SEND_TYPE_DEFAULT,
    * and the default for responses is to match the request.
    */
   private int sendType = Attachments.SEND_TYPE_NOTSET;

   /**
    * Our ServiceDescription.  Holds pretty much all the interesting
    * metadata about this service.
    */
   private ServiceDesc serviceDescription = new ServiceDesc();
   private AxisEngine engine;

   /**
    * Actor list - these are just the service-specific ones
    */
   ArrayList actors = new ArrayList();

   /**
    * Get the service-specific actor list
    *
    * @return
    */
   public ArrayList getServiceActors()
   {
      return actors;
   }

   /**
    * Get the merged actor list for this service, including engine-wide
    * actor URIs.
    *
    * @return
    */
   public ArrayList getActors()
   {
      ArrayList acts = (ArrayList)actors.clone();  // ??? cache this?

      // TODO: a SOAPService should always be associated with an engine,
      // so this should never be null.... check all paths to ensure that
      // constraint is true.
      if (engine != null)
      {
         acts.addAll(engine.getActorURIs());
      }
      return acts;
   }


   /**
    * SOAPResponseHandler is used to inject SOAP semantics just before
    * the pivot handler.
    */
   private class SOAPResponseHandler extends BasicHandler
   {
      public SOAPResponseHandler()
      {
      }

      public void invoke(MessageContext msgContext) throws AxisFault
      {
         // Do SOAP semantics here
         if (log.isDebugEnabled())
         {
            log.debug(Messages.getMessage("semanticCheck00"));
         }

         checkMustUnderstand(msgContext);
      }

      /**
       * Check whether we should throw MustUnderstand fault. Now, if there is an actor that is not
       * service actor and there is a MustUnderstand flat to 1, we will need to throw the fault exception
       * since we don't know how to handle it.
       *
       * @param msgContext
       * @throws AxisFault
       */
      private void checkMustUnderstand(MessageContext msgContext) throws AxisFault
      {
         String svcActor = ""; // can't handle other actor for now.

         // 1. Check mustUnderstands
         SOAPEnvelopeAxisImpl env = msgContext.getRequestMessage().getSOAPEnvelope();
         Vector headers = env.getHeaders();  // TODO get only headers with actors.
         Vector misunderstoodHeaders = null;
         Enumeration en = headers.elements();
         while (en.hasMoreElements())
         {
            SOAPHeaderElementAxisImpl header = (SOAPHeaderElementAxisImpl)en.
                    nextElement();
/*
                if (header.getMustUnderstand() && !header.isProcessed()) {
                    if (misunderstoodHeaders == null)
                        misunderstoodHeaders = new Vector();
                    misunderstoodHeaders.addElement(header);
                }
*/
            if (header.getActor() != null && header.getMustUnderstand() &&
                    header.getActor() != svcActor)
            {
               if (misunderstoodHeaders == null)
                  misunderstoodHeaders = new Vector();
               misunderstoodHeaders.addElement(header);
            }
         }

         SOAPConstants soapConstants = msgContext.getSOAPConstants();
         // !!! we should indicate SOAP1.2 compliance via the
         // MessageContext, not a boolean here....

         if (misunderstoodHeaders != null)
         {
            AxisFault fault =
                    new AxisFault(soapConstants.getMustunderstandFaultQName(),
                            null, null,
                            null, null,
                            null);

            StringBuffer whatWasMissUnderstood = new StringBuffer(256);

            // !!! If SOAP 1.2, insert misunderstood fault headers here
            if (soapConstants == SOAPConstants.SOAP12_CONSTANTS)
            {
               en = misunderstoodHeaders.elements();
               while (en.hasMoreElements())
               {
                  SOAPHeaderElementAxisImpl badHeader = (SOAPHeaderElementAxisImpl)en.
                          nextElement();
                  QName badQName = new QName(badHeader.getNamespaceURI(),
                          badHeader.getName());

                  if (whatWasMissUnderstood.length() != 0) whatWasMissUnderstood.append(", ");
                  whatWasMissUnderstood.append(badQName.toString());

                  SOAPHeaderElementAxisImpl newHeader = new
                          SOAPHeaderElementAxisImpl(Constants.URI_SOAP12_ENV,
                                  Constants.ELEM_NOTUNDERSTOOD);
                  newHeader.addAttribute(null,
                          Constants.ATTR_QNAME,
                          badQName);

                  fault.addHeader(newHeader);
               }
            }

            fault.setFaultString(Messages.getMessage("noUnderstand00",
                    whatWasMissUnderstood.toString()));

            throw fault;
         }
      }
   }

   /**
    * Standard, no-arg constructor.
    */
   public SOAPService()
   {
      setOptionsLockable(true);
      initHashtable();

      // For now, always assume we're the ultimate destination.
      actors.add("");
   }

   /**
    * Constructor with real or null request, pivot, and response
    * handlers. A special request handler is specified to inject
    * SOAP semantics.
    */
   public SOAPService(Handler reqHandler, Handler pivHandler,
                      Handler respHandler)
   {
      this();
      init(reqHandler, null, pivHandler, new SOAPResponseHandler(), respHandler);
   }

   public TypeMappingRegistry getTypeMappingRegistry()
   {
      return serviceDescription.getTypeMappingRegistry();
   }

   /**
    * Convenience constructor for wrapping SOAP semantics around
    * "service handlers" which actually do work.
    */
   public SOAPService(Handler serviceHandler)
   {
      init(null, null, serviceHandler, new SOAPResponseHandler(), null);
   }

   /**
    * Tell this service which engine it's deployed to.
    */
   public void setEngine(AxisEngine engine)
   {
      if (engine == null)
         throw new IllegalArgumentException(Messages.getMessage("nullEngine"));

      this.engine = engine;
      getTypeMappingRegistry().delegate(engine.getTypeMappingRegistry());
   }

   public AxisEngine getEngine()
   {
      return engine;
   }

   public boolean availableFromTransport(String transportName)
   {
      if (validTransports != null)
      {
         for (int i = 0; i < validTransports.size(); i++)
         {
            if (validTransports.elementAt(i).equals(transportName))
               return true;
         }
         return false;
      }

      return true;
   }

   public Style getStyle()
   {
      return serviceDescription.getStyle();
   }

   public void setStyle(Style style)
   {
      serviceDescription.setStyle(style);
   }

   public Use getUse()
   {
      return serviceDescription.getUse();
   }

   public void setUse(Use style)
   {
      serviceDescription.setUse(style);
   }

   public ServiceDesc getServiceDescription()
   {
      return serviceDescription;
   }

   /**
    * Returns a service description with the implementation class filled in.
    * Syncronized to prevent simutaneous modification of serviceDescription.
    */
   public synchronized ServiceDesc getInitializedServiceDesc(MessageContext msgContext)
           throws AxisFault
   {

      if (serviceDescription.getImplClass() == null)
      {

         // Let the provider do the work of filling in the service
         // descriptor.  This is so that it can decide itself how best
         // to map the Operations.  In the future, we may want to support
         // providers which don't strictly map to Java class backends
         // (BSFProvider, etc.), and as such we hand off here.
         if (pivotHandler instanceof BasicProvider)
         {
            ((BasicProvider)pivotHandler).initServiceDesc(this, msgContext);
         }

      }

      return serviceDescription;
   }

   public void setServiceDescription(ServiceDesc serviceDescription)
   {
      if (serviceDescription == null)
      {
         // FIXME: Throw NPE?
         return;
      }
      this.serviceDescription = serviceDescription;
   }

   public void setPropertyParent(Hashtable parent)
   {
      if (options == null)
      {
         options = new LockableHashtable();
      }
      ((LockableHashtable)options).setParent(parent);
   }

   /**
    * Generate WSDL.  If we have a specific file configured in the
    * ServiceDesc, just return that.  Otherwise run through all the Handlers
    * (including the provider) and call generateWSDL() on them via our
    * parent's implementation.
    */
   public void generateWSDL(MessageContext msgContext) throws AxisFault
   {
      if (serviceDescription == null ||
              serviceDescription.getWSDLFile() == null)
      {
         super.generateWSDL(msgContext);
         return;
      }
      InputStream instream = null;

      // Got a WSDL file in the service description, so try and read it
      try
      {
         String filename = serviceDescription.getWSDLFile();
         File file = new File(filename);
         if (file.exists())
         {
            //if this resolves to a file, load it
            instream = new FileInputStream(filename);
         }
         else
         {
            //else load a named resource in our classloader.
            ClassLoader classLoader = msgContext.getClassLoader();
            if (classLoader == null)
               classLoader = getClass().getClassLoader();

            instream = classLoader.getResourceAsStream(filename);
            if (instream == null)
            {
               String errorText = Messages.getMessage("wsdlFileMissing", filename);
               throw new AxisFault(errorText);
            }
         }
         Document doc = XMLUtils.newDocument(instream);
         msgContext.setProperty("WSDL", doc);
      }
      catch (Exception e)
      {
         throw AxisFault.makeFault(e);
      }
      finally
      {
         if (instream != null)
         {
            try
            {
               instream.close();
            }
            catch (IOException e)
            {
            }
         }
      }
   }
   /*********************************************************************
    * Administration and management APIs
    *
    * These can get called by various admin adapters, such as JMX MBeans,
    * our own Admin client, web applications, etc...
    *
    *********************************************************************
    */

   /**
    * Placeholder for "enable this service" method
    */
   public void start()
   {
   }

   /**
    * Placeholder for "disable this service" method
    */
   public void stop()
   {
   }

   /**
    * Make this service available on a particular transport
    */
   public void enableTransport(String transportName)
   {
      if (log.isDebugEnabled())
      {
         log.debug(Messages.getMessage("enableTransport00", "" + this, transportName));
      }

      if (validTransports == null)
         validTransports = new Vector();
      validTransports.addElement(transportName);
   }

   /**
    * Disable access to this service from a particular transport
    */
   public void disableTransport(String transportName)
   {
      if (validTransports != null)
      {
         validTransports.removeElement(transportName);
      }
   }

   public boolean needsHighFidelityRecording()
   {
      return highFidelityRecording;
   }

   public void setHighFidelityRecording(boolean highFidelityRecording)
   {
      this.highFidelityRecording = highFidelityRecording;
   }

   // see org.jboss.axis.attachments.Attachments
   public int getSendType()
   {
      return sendType;
   }

   public void setSendType(int sendType)
   {
      this.sendType = sendType;
   }

   public void invoke(MessageContext msgContext) throws AxisFault
   {
      HandlerInfoChainFactory handlerFactory = (HandlerInfoChainFactory)this.getOption(Constants.ATTR_HANDLERINFOCHAIN);
      HandlerChainImpl handlerImpl = null;
      if (handlerFactory != null) handlerImpl = (HandlerChainImpl)handlerFactory.createHandlerChain();
      boolean result = true;

      if (handlerImpl != null)
      {
         result = handlerImpl.handleRequest(msgContext);
      }

      if (result)
      {
         super.invoke(msgContext);
      }
      else
      {
         msgContext.setPastPivot(true);
      }

      if (handlerImpl != null)
      {
         handlerImpl.handleResponse(msgContext);
         handlerImpl.destroy();
      }
   }
}
