/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.axis.attachments;

// $Id$

import org.jboss.axis.AxisEngine;
import org.jboss.axis.MessageContext;
import org.jboss.axis.utils.ClassUtils;
import org.jboss.axis.utils.Messages;
import org.jboss.logging.Logger;

/**
 * Provide access to the attechment support implementation.
 * <p/>
 * The default implementation is {@link AttachmentsImpl}
 * You can overwrite the implementation through the engine option {@link AxisEngine.PROP_ATTACHMENT_IMPLEMENTATION}
 *
 * @author Thomas Diesler (thomas.diesler@jboss.org)
 * @since 31-May-2004
 */
public final class AttachmentSupport
{

   /**
    * Provides logging
    */
   private static Logger log = Logger.getLogger(AttachmentSupport.class.getName());

   /**
    * Default Attachments Implementation class.
    */
   public static final String DEFAULT_ATTACHMNET_IMPL = AxisEngine.DEFAULT_ATTACHMENT_IMPL;

   /**
    * Current Attachment implementation.
    */
   private static String attachImplClassName = DEFAULT_ATTACHMNET_IMPL;

   private static boolean checkForAttachmentSupport = true;
   private static boolean attachmentSupportEnabled = false;
   private static Class attachImplClass;

   /**
    * Check if we have attachments enabled.
    * <p/>
    * The check is only done once.
    */
   public static synchronized boolean isAttachmentSupportEnabled()
   {
      return isAttachmentSupportEnabled(null);
   }

   /**
    * Check if we have attachments enabled.
    * <p/>
    * The check is only done once.
    */
   public static synchronized boolean isAttachmentSupportEnabled(MessageContext msgCtx)
   {

      if (checkForAttachmentSupport)
      {
         checkForAttachmentSupport = false;

         try
         {
            if (msgCtx != null)
            {
               AxisEngine engine = msgCtx.getAxisEngine();
               if (engine != null)
               {
                  attachImplClassName = (String)engine.getOption(AxisEngine.PROP_ATTACHMENT_IMPLEMENTATION);
               }
            }

            if (attachImplClassName == null)
            {
               attachImplClassName = AxisEngine.DEFAULT_ATTACHMENT_IMPL;
            }

            /**
             * Attempt to resolve class name, verify that these are present...
             */
            ClassUtils.forName("javax.activation.DataHandler");
            ClassUtils.forName("javax.mail.internet.MimeMultipart");

            attachImplClass = ClassUtils.forName(attachImplClassName);
            attachmentSupportEnabled = true;
         }
         catch (ClassNotFoundException ex)
         {
            // no support for it, leave mAttachments null.
         }
         catch (NoClassDefFoundError ex)
         {
            // no support for it, leave mAttachments null.
         }
         log.debug(Messages.getMessage("attachEnabled") + "  " + attachmentSupportEnabled);
      }

      return attachmentSupportEnabled;
   }

   /**
    * Get the attachment implementation class
    */
   public static Class getImplementationClass()
   {
      return attachImplClass;
   }
}
