/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.ejb3.stateless;


import java.lang.reflect.Method;
import java.util.Hashtable;
import java.util.Map;
import javax.ejb.EJBException;
import javax.ejb.Timer;
import javax.ejb.TimerService;
import org.jboss.annotation.ejb.LocalBinding;
import org.jboss.annotation.ejb.RemoteBinding;
import org.jboss.annotation.ejb.RemoteBindings;
import org.jboss.aop.AspectManager;
import org.jboss.aop.MethodInfo;
import org.jboss.aop.advice.Interceptor;
import org.jboss.aop.joinpoint.Invocation;
import org.jboss.aop.joinpoint.InvocationResponse;
import org.jboss.aop.joinpoint.MethodInvocation;
import org.jboss.aop.util.MethodHashing;
import org.jboss.aspects.asynch.FutureHolder;
import org.jboss.ejb.AllowedOperationsAssociation;
import org.jboss.ejb.AllowedOperationsFlags;
import org.jboss.ejb.txtimer.TimedObjectInvoker;
import org.jboss.ejb3.EJBContainerInvocation;
import org.jboss.ejb3.Ejb3Deployment;
import org.jboss.ejb3.EjbTimerUtil;
import org.jboss.ejb3.ProxyUtils;
import org.jboss.ejb3.SessionContainer;
import org.jboss.ejb3.interceptor.InterceptorInfoRepository;
import org.jboss.logging.Logger;
import org.jboss.proxy.ejb.handle.HomeHandleImpl;
import org.jboss.proxy.ejb.handle.StatelessHandleImpl;


/**
 * Comment
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @version $Revision$
 */
public class StatelessContainer extends SessionContainer implements TimedObjectInvoker
{
   private static final Logger log = Logger.getLogger(StatelessContainer.class);

   protected TimerService timerService;

   public StatelessContainer(ClassLoader cl, String beanClassName, String ejbName, AspectManager manager,
                             Hashtable ctxProperties, InterceptorInfoRepository interceptorRepository,
                             Ejb3Deployment deployment)
   {
      super(cl, beanClassName, ejbName, manager, ctxProperties, interceptorRepository, deployment);
      beanContextClass = StatelessBeanContext.class;
   }

   public void start() throws Exception
   {
      try
      {
         super.start();
         timerService = EjbTimerUtil.getTimerService(this, this);
      }
      catch (Exception e)
      {
         try
         {
            stop();
         }
         catch (Exception ignore)
         {
            log.debug("Failed to cleanup after start() failure", ignore);
         }
         throw e;
      }
   }

   public void stop() throws Exception
   {
      if (timerService != null) EjbTimerUtil.removeTimerService(this);
      super.stop();
   }

   public TimerService getTimerService()
   {
      return timerService;
   }

   public void callTimeout(Timer timer) throws Exception
   {
      Method timeout = callbackHandler.getTimeoutCallback();
      if (timeout == null) throw new EJBException("No method has been annotated with @Timeout");
      Object[] args = {timer};
      ClassLoader oldLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         AllowedOperationsAssociation.pushInMethodFlag(AllowedOperationsFlags.IN_EJB_TIMEOUT);
         try
         {
            MethodInfo info = (MethodInfo) methodInterceptors.get(callbackHandler.getTimeoutCalllbackHash());
            Interceptor[] aspects = info.interceptors;
            EJBContainerInvocation nextInvocation = new EJBContainerInvocation(info, aspects);
            nextInvocation.setAdvisor(this);
            nextInvocation.setArguments(args);
            nextInvocation.invokeNext();
         }
         catch (Throwable throwable)
         {
            if (throwable instanceof Exception) throw (Exception) throwable;
            throw new RuntimeException(throwable);
         }
         finally
         {
            AllowedOperationsAssociation.popInMethodFlag();
         }
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(oldLoader);
      }
   }

   /**
    * Performs a synchronous local invocation
    */
   public Object localInvoke(Method method, Object[] args) throws Throwable
   {
      return localInvoke(method, args, null);
   }

   /**
    * Performs a synchronous or asynchronous local invocation
    *
    * @param provider If null a synchronous invocation, otherwise an asynchronous
    */
   public Object localInvoke(Method method, Object[] args, FutureHolder provider) throws Throwable
   {
      ClassLoader oldLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         long hash = MethodHashing.calculateHash(method);
         MethodInfo info = (MethodInfo) methodInterceptors.get(hash);
         if (info == null)
         {
            throw new RuntimeException("Could not resolve beanClass method from proxy call: " + method.toString());
         }
         
         if (info.unadvisedMethod != null)
            invokedMethod.put(this, info.unadvisedMethod);

         if (info.unadvisedMethod != null && isHomeMethod(info.unadvisedMethod))
         {
            return invokeLocalHomeMethod(info, args);
         }

         Interceptor[] aspects = info.interceptors;
         EJBContainerInvocation nextInvocation = new EJBContainerInvocation(info, aspects);
         nextInvocation.setAdvisor(this);
         nextInvocation.setArguments(args);

         ProxyUtils.addLocalAsynchronousInfo(nextInvocation, provider);
         return nextInvocation.invokeNext();
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(oldLoader);
      }
   }

   public InvocationResponse dynamicInvoke(Object target, Invocation invocation) throws Throwable
   {
      ClassLoader oldLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(classloader);
         MethodInvocation si = (MethodInvocation) invocation;
         MethodInfo info = (MethodInfo) methodInterceptors.get(si.getMethodHash());
         if (info == null)
         {
            throw new RuntimeException("Could not resolve beanClass method from proxy call");
         }
         
         if (info.unadvisedMethod != null)
            invokedMethod.put(this, info.unadvisedMethod);

         Map responseContext = null;
         Object rtn = null;
         if (info.unadvisedMethod != null && isHomeMethod(info.unadvisedMethod))
         {
            rtn = invokeHomeMethod(info, si);
         }
         else if (info != null && info.unadvisedMethod != null && isEJBObjectMethod(info.unadvisedMethod))
         {
            rtn = invokeEJBObjectMethod(info, si);
         }
         else
         {

            EJBContainerInvocation newSi = null;
            Interceptor[] aspects = info.interceptors;

            newSi = new EJBContainerInvocation(info, aspects);
            newSi.setArguments(si.getArguments());
            newSi.setMetaData(si.getMetaData());
            newSi.setAdvisor(this);
            rtn = newSi.invokeNext();
            responseContext = newSi.getResponseContextInfo();
         }

         InvocationResponse response = marshallResponse(invocation, rtn, responseContext);
         return response;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(oldLoader);
      }

   }

   protected Object invokeEJBObjectMethod(MethodInfo info, MethodInvocation invocation) throws Throwable
   {
      if (info.unadvisedMethod.getName().equals("getHandle"))
      {
         StatelessHandleImpl handle = null;
         RemoteBinding remoteBindingAnnotation = (RemoteBinding) resolveAnnotation(RemoteBinding.class);
         if (remoteBindingAnnotation != null)
            handle = new StatelessHandleImpl(remoteBindingAnnotation.jndiBinding());

         return handle;
      }
      else if (info.unadvisedMethod.getName().equals("remove"))
      {
         return null;
      }
      else if (info.unadvisedMethod.getName().equals("getEJBHome"))
      {
         HomeHandleImpl homeHandle = null;

         RemoteBinding remoteBindingAnnotation = (RemoteBinding) resolveAnnotation(RemoteBinding.class);
         if (remoteBindingAnnotation != null)
            homeHandle = new HomeHandleImpl(remoteBindingAnnotation.jndiBinding() + "Home");

         return homeHandle.getEJBHome();
      }
      else if (info.unadvisedMethod.getName().equals("getPrimaryKey"))
      {
         return null;
      }
      else if (info.unadvisedMethod.getName().equals("isIdentical"))
      {
         return false;
      }
      else
      {
         return null;
      }
   }

   private Object invokeLocalHomeMethod(MethodInfo info, Object[] args) throws Exception
   {
      if (info.unadvisedMethod.getName().equals("create"))
      {
         LocalBinding binding = (LocalBinding) resolveAnnotation(LocalBinding.class);

         StatelessLocalProxyFactory factory = new StatelessLocalProxyFactory();
         factory.setContainer(this);
         factory.init();

         Object proxy = factory.createProxy();

         return proxy;
      }
      else // remove
      {
         return null;
      }
   }

   protected Object invokeHomeMethod(MethodInfo info, MethodInvocation invocation) throws Throwable
   {
      if (info.unadvisedMethod.getName().equals("create"))
      {
         RemoteBinding binding = null;

         RemoteBindings bindings = (RemoteBindings) resolveAnnotation(RemoteBindings.class);
         if (bindings != null)
            binding = bindings.value()[0];
         else
            binding = (RemoteBinding) resolveAnnotation(RemoteBinding.class);

         StatelessRemoteProxyFactory factory = new StatelessRemoteProxyFactory();
         factory.setContainer(this);
         factory.setRemoteBinding(binding);
         factory.init();

         return factory.createProxy();
      }
      else // remove
      {
         return null;
      }
   }
   
   public Object getInvokedBusinessInterface(Object key) throws IllegalStateException
   {
      Method method = (Method)invokedMethod.get(this);
      
      if (method == null)
         throw new IllegalStateException("No invocation");
      
      return getInvokedInterface(method);
   }
   
   public Object getBusinessObject(Object key, Class businessInterface) throws IllegalStateException
   {
      Method method = (Method)invokedMethod.get(this);
      
      if (method == null)
         throw new IllegalStateException("No invocation");
      
      Object invokedInterface = getInvokedInterface(method);
      
      if (!invokedInterface.equals(businessInterface))
         throw new IllegalStateException("Business interface " + businessInterface + " was not invoked interface");
      
      return invokedInterface;
   }
}
