/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ejb3.injection;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import javax.naming.InitialContext;
import org.jboss.annotation.JndiInject;
import org.jboss.ejb3.Container;
import org.jboss.ejb3.EJBContainer;
import org.jboss.ejb3.interceptor.InterceptorInjector;
import org.jboss.logging.Logger;

/**
 * Searches bean class for all @Inject and create Injectors
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @version $Revision$
 */
public class JndiInjectHandler
{
   private static final Logger log = Logger.getLogger(JndiInjectHandler.class);
   
   public static List loadInjectors(Container container) throws Exception
   {
      Class clazz = container.getBeanClass();
      return loadInjectors(clazz, (EJBContainer)container, true);
   }

   public static List loadInjectors(InterceptorInjector injector) throws Exception
   {
      return loadInjectors(injector.getClazz(), (EJBContainer)injector.getContainer(), false);
   }
   
   private static List loadInjectors(Class clazz, EJBContainer container, boolean isContainer) throws Exception
   {
      ArrayList list = new ArrayList();
      HashSet<String> visitedMethods = new HashSet<String>();
      loadMethodInjectors(visitedMethods, clazz, container, list, isContainer);
      loadFieldInjectors(clazz, container, list, isContainer);
      return list;
   }
   
   private static void loadMethodInjectors(HashSet<String> visitedMethods, Class clazz, EJBContainer container, ArrayList list, boolean isContainer) throws Exception
   {
      if (clazz == null || clazz.equals(Object.class))
      {
         return;
      }
      Method[] methods = clazz.getDeclaredMethods();
      InitialContext ctx = container.getInitialContext();
      for (int i = 0; i < methods.length; i++)
      {
         JndiInject ref = (JndiInject)InjectionUtil.getAnnotation(
               JndiInject.class, container, methods[i], isContainer); 
         if (ref != null)
         {
            if (!Modifier.isPrivate(methods[i].getModifiers()))
            {
               if (visitedMethods.contains(methods[i].getName())) continue;
               else visitedMethods.add(methods[i].getName());
            }
            if (!methods[i].getName().startsWith("set")) throw new RuntimeException("@JndiInject can only be used with a set method: " + methods[i]);
            if (methods[i].getParameterTypes().length != 1) throw new RuntimeException("@JndiInject can only be used with a set method of one parameter: " + methods[i]);
            String jndiName = methods[i].getName().substring(3);
            if (ref.jndiName() != null && !ref.jndiName().equals(""))
            {
               jndiName = ref.jndiName();
            }
            list.add(new JndiMethodInjector(methods[i], jndiName, ctx));
         }
      }
      if (clazz != null && !clazz.equals(Object.class))
      {
         loadMethodInjectors(visitedMethods, clazz.getSuperclass(), container, list, isContainer);
      }

   }
   
   private static void loadFieldInjectors(Class clazz, EJBContainer container, ArrayList list, boolean isContainer) throws Exception
   {
      if (clazz == null || clazz.equals(Object.class)) return;
      loadFieldInjectors(clazz.getSuperclass(), container, list, isContainer);
      Field[] fields = clazz.getDeclaredFields();
      InitialContext ctx = container.getInitialContext();
      for (int i = 0; i < fields.length; i++)
      {
         JndiInject ref = (JndiInject)InjectionUtil.getAnnotation(
               JndiInject.class, container, fields[i], isContainer); 
         if (ref != null)
         {
            String jndiName = fields[i].getName();
            if (ref.jndiName() != null && !ref.jndiName().equals(""))
            {
               jndiName = ref.jndiName();
            }
            list.add(new JndiFieldInjector(fields[i], jndiName, ctx));
         }
      }
   }
}
