/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ejb3.cache.tree;

import javax.ejb.EJBException;
import javax.ejb.EJBNoSuchObjectException;
import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.jboss.annotation.ejb.cache.tree.CacheConfig;
import org.jboss.aop.Advisor;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.TreeCacheMBean;
import org.jboss.ejb3.Container;
import org.jboss.ejb3.Pool;
import org.jboss.ejb3.cache.ClusteredStatefulCache;
import org.jboss.ejb3.stateful.StatefulBeanContext;
import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.mx.util.MBeanServerLocator;

/**
 * Comment
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @version $Revision$
 */
public class StatefulTreeCache implements ClusteredStatefulCache
{
   private Pool pool;
   private PassivationTreeCache cache;
   private String cacheNode;
   private StatefulEvictionPolicy eviction;

   public StatefulBeanContext create()
   {
      StatefulBeanContext ctx = null;
      try
      {
         ctx = (StatefulBeanContext) pool.get();
         cache.put(cacheNode + "/" + ctx.getId(), "bean", ctx);
      }
      catch (Exception e)
      {
         throw new EJBException(e);
      }
      return ctx;
   }

   public StatefulBeanContext create(Class[] initTypes, Object[] initValues)
   {
      StatefulBeanContext ctx = null;
      try
      {
         ctx = (StatefulBeanContext) pool.get(initTypes, initValues);
         cache.put(cacheNode + "/" + ctx.getId(), "bean", ctx);
      }
      catch (Exception e)
      {
         throw new EJBException(e);
      }
      return ctx;
   }

   public StatefulBeanContext get(Object key) throws EJBException
   {
      StatefulBeanContext entry = null;
      try
      {
         Object obj = cache.get(cacheNode + "/" + key, "bean");
         //System.out.println("obj: instanceof: " + obj.getClass().getName());
         entry = (StatefulBeanContext) obj;
      }
      catch (CacheException e)
      {
         throw new RuntimeException(e);
      }
      if (entry == null)
      {
         throw new EJBNoSuchObjectException("Could not find Stateful bean: " + key);
      }
      entry.inUse = true;
      entry.lastUsed = System.currentTimeMillis();
      return entry;
   }

   public void remove(Object key)
   {
      StatefulBeanContext ctx = null;
      try
      {
         cache.remove(cacheNode + "/" + key);
      }
      catch (CacheException e)
      {
         throw new RuntimeException(e);
      }
      if (ctx != null) pool.remove(ctx);

   }

   public void finished(StatefulBeanContext ctx)
   {
      synchronized (ctx)
      {
         ctx.inUse = false;
         ctx.lastUsed = System.currentTimeMillis();
         if (ctx.markedForPassivation)
         {
            try
            {
               Fqn fqn = Fqn.fromString(cacheNode + "/" + ctx.getId());
               eviction.passivate(fqn, ctx);
            }
            catch (Exception e)
            {
               throw new RuntimeException(e);
            }
         }
      }
   }

   public void replicate(StatefulBeanContext ctx)
   {
      try
      {
         cache.put(cacheNode + "/" + ctx.getId(), "bean", ctx);
      }
      catch (CacheException e)
      {
         throw new RuntimeException(e);
      }
   }

   public void initialize(Container container) throws Exception
   {
      Advisor advisor = (Advisor) container;
      this.pool = container.getPool();
      CacheConfig config = (CacheConfig) advisor.resolveAnnotation(CacheConfig.class);
      MBeanServer server = MBeanServerLocator.locateJBoss();
      ObjectName cacheON = new ObjectName(config.name());
      TreeCacheMBean mbean = (TreeCacheMBean) MBeanProxyExt.create(TreeCacheMBean.class, cacheON, server);
      cache = (PassivationTreeCache) mbean.getInstance();
      cacheNode = "/" + container.getEjbName();
      eviction = (StatefulEvictionPolicy) cache.getEvictionPolicy();
      eviction.createRegion(cacheNode, config.maxSize(), config.idleTimeoutSeconds());
   }

   public void start()
   {

   }

   public void stop()
   {
   }
}
