/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.xml.binding;

// $Id$

import javax.xml.namespace.QName;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * A simple namespace registry
 *
 * @author thomas.diesler@jboss.org
 * @since 08-June-2004
 */
public class NamespaceRegistry
{
   public static final String PREFIX_XML_SCHEMA = "xsd";
   public static final String URI_XML_SCHEMA = "http://www.w3.org/2001/XMLSchema";
   
   public static final String PREFIX_XML_SCHEMA_INSTANCE = "xsi";
   public static final String URI_XML_SCHEMA_INSTANCE = "http://www.w3.org/2000/10/XMLSchema-instance";

   // Maps namespace uri to prefix
   private HashMap namespaceMap = new HashMap();
   private int namespaceIndex;

   public NamespaceRegistry()
   {
      namespaceMap.put(URI_XML_SCHEMA, PREFIX_XML_SCHEMA);
      namespaceMap.put(URI_XML_SCHEMA_INSTANCE, PREFIX_XML_SCHEMA_INSTANCE);
   }

   /** Register a QName and return a QName that is guarantied to have a prefix
    */
   public QName registerQName(QName qname)
   {
      if (qname == null)
         return null;

      String prefix = registerNamespaceURI(qname.getNamespaceURI(), qname.getPrefix());
      qname = new QName(qname.getNamespaceURI(), qname.getLocalPart(), prefix);
      return qname;
   }

   /** Register the given namespaceURI/prefix combination.
    * In case the prefix is null, it will be assigend.
    *
    * @param namespaceURI The namespaceURI
    * @param prefix The corresponding prefix, maybe null
    * @return A prefix, never null
    */
   public String registerNamespaceURI(String namespaceURI, String prefix)
   {
      String regPrefix = (String)namespaceMap.get(namespaceURI);

      if (regPrefix == null && prefix != null && prefix.length() > 0)
         regPrefix = prefix;

      if (regPrefix == null)
         regPrefix = "ns" + (++namespaceIndex);

      namespaceMap.put(namespaceURI, regPrefix);
      return regPrefix;
   }

   public String getPrefix(String namespaceURI)
   {
      return (String)namespaceMap.get(namespaceURI);
   }

   public String getNamespaceURI(String prefix)
   {
      Iterator it = namespaceMap.entrySet().iterator();
      while (it.hasNext())
      {
         Map.Entry entry = (Map.Entry)it.next();
         if (prefix.equals(entry.getValue()))
            return (String)entry.getKey();
      }
      return null;
   }

   public Iterator getRegisteredNamespaceURIs()
   {
      return namespaceMap.keySet().iterator();
   }
}
