/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
 
package org.jboss.monitor.services;

import javax.management.Notification;

import EDU.oswego.cs.dl.util.concurrent.SynchronizedLong;

import org.jboss.system.ListenerServiceMBeanSupport;

import org.jboss.monitor.alarm.Alarm;
import org.jboss.monitor.alarm.AlarmTable;
import org.jboss.monitor.alarm.AlarmTableNotification;
import org.jboss.monitor.alarm.MBeanImplAccess;

/**
 * ActiveAlarmTable
 *
 * @jmx:mbean
 *    extends="org.jboss.system.ListenerServiceMBean"
 * 
 * @author  <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 *
 * @version $Revision:1$
**/
public class ActiveAlarmTable
    extends ListenerServiceMBeanSupport
    implements ActiveAlarmTableMBean
{
   // Private Data --------------------------------------------------
    
   /** Number of processed JMX notifications */
   private SynchronizedLong notificationCount;
   
   /** alarm table */
   AlarmTable almtab =
      new AlarmTable(
         new MBeanImplAccess() {
            public javax.management.ObjectName getMBeanName() { return getServiceName(); }
            public long getSequenceNumber() { return getNextNotificationSequenceNumber(); }
            public void emitNotification(javax.management.Notification n) { sendNotification(n); }
         }
      );
      
   // Constructors --------------------------------------------------
   
   /**
    * CTOR
   **/
   public ActiveAlarmTable()
   {
      this.notificationCount = new SynchronizedLong(0);
      this.almtab.setServerId(Alarm.DEFAULT_SERVER_ID);
   }
   
   // Attributes ----------------------------------------------------
   
   /**
    * Number of notifications received.
    *
    * @jmx:managed-attribute
   **/
   public long getNotificationsProcessed()
   {
      return this.notificationCount.get();
   }

   /**
    * Sets the serverId to use in the produced AlarmTableNotifications
    *
    * @jmx:managed-attribute
   **/
   public void setServerId(String serverId)
   {
      if (serverId != null)
         this.almtab.setServerId(serverId);
   }
   
   /**
    * Gets the serverId
    *
    * @jmx:managed-attribute    
   **/
   public String getServerId()
   {
      return this.almtab.getServerId();
   }
   
   // Operations ----------------------------------------------------

   /**
    * Acknowledge all
    *
    * @jmx:managed-operation
    *
    * @return number of acknowledged alarms       
   **/
   public int acknowledgeAll(String user, String system)
   {
      AlarmTableNotification[] atns = this.almtab.getAlarmTable();
      int acked = 0;
      
      for (int i = 0; i < atns.length; ++i) {
         AlarmTableNotification atn = atns[i];
         
         if (this.almtab.acknowledge(atn.createKey(), user, system))
            ++acked;
      }
      return acked;
   }

   /**
    * Uncknowledge all
    *
    * @jmx:managed-operation
    *
    * @return number of unacknowledged alarms       
   **/
   public int unacknowledgeAll(String user, String system)
   {
      AlarmTableNotification[] atns = this.almtab.getAlarmTable();
      int unacked = 0;
      
      for (int i = 0; i < atns.length; ++i) {
         AlarmTableNotification atn = atns[i];
         
         if (this.almtab.unacknowledge(atn.createKey(), user, system))
            ++unacked;
      }
      return unacked;
   }
   
   /**
    * Acknowledge an Alarm
    *
    * @jmx:managed-operation   
    *
    * @return true if ack was succesful, false otherwise
    *         (not in table or acked already)
   **/
   public boolean acknowledge(String serverId, String source, String type,
                              String user, String system)
   {
      return this.almtab.acknowledge(serverId, source, type, user, system);
   }

   /**
    * Unacknowledge an Alarm
    *
    * @jmx:managed-operation   
    *
    * @return true if unack was succesful, false otherwise
    *         (not in table or unacked already)
   **/
   public boolean unacknowledge(String serverId, String source, String type,
                                String user, String system)
   {
      return this.almtab.unacknowledge(serverId, source, type, user, system);
   }
   
   /**
    * Gets the ActiveAlarmTable
    *
    * @jmx:managed-operation    
   **/
   public AlarmTableNotification[] fetchAlarmTable()
   {
      return this.almtab.getAlarmTable();
   }
   
   /**
    * Gets the ActiveAlarmTable as Html
    *
    * @jmx:managed-operation    
   **/
   public String fetchAlarmTableAsHtml()
   {
      AlarmTableNotification[] tab = this.almtab.getAlarmTable();
      
      StringBuffer sbuf = new StringBuffer(8192);
      
      sbuf.append("<table border=\"1\">").append("\n");
      
      sbuf.append("<tr>");
      sbuf.append("<td></td>");
      sbuf.append("<td>serverId</td>");
      sbuf.append("<td>source</td>");
      sbuf.append("<td>type</td>");
      sbuf.append("<td>alarmState</td>");
      sbuf.append("<td>severity</td>");
      sbuf.append("<td>timeStamp</td>");
      sbuf.append("<td>sequenceNumber</td>");
      sbuf.append("<td>message</td>");
      sbuf.append("<td>userData</td>");
      sbuf.append("<td>ackState</td>");
      sbuf.append("<td>ackTime</td>");
      sbuf.append("<td>ackUser</td>");
      sbuf.append("<td>ackSystem</td");
      sbuf.append("</tr>").append("\n");
      
      for (int i = 0; i <  tab.length; i++) {
         AlarmTableNotification atn = tab[i];
         Notification n = (Notification)atn.getUserData();
         
         sbuf.append("<tr>");
         sbuf.append("<td>").append(i).append("</td>");
         sbuf.append("<td>").append(atn.getServerId()).append("</td>");
         sbuf.append("<td>").append(n.getSource()).append("</td>");
         sbuf.append("<td>").append(n.getType()).append("</td>");
         sbuf.append("<td>").append(Alarm.STATE_STRINGS[atn.getAlarmState()]).append("</td>");
         sbuf.append("<td>").append(Alarm.SEVERITY_STRINGS[atn.getSeverity()]).append("</td>");
         sbuf.append("<td>").append(n.getTimeStamp()).append("</td>");
         sbuf.append("<td>").append(n.getSequenceNumber()).append("</td>");
         sbuf.append("<td>").append(n.getMessage()).append("</td>");
         sbuf.append("<td>").append(n.getUserData()).append("</td>");
         sbuf.append("<td>").append(atn.getAckState()).append("</td>");
         sbuf.append("<td>").append(atn.getAckTime()).append("</td>");
         sbuf.append("<td>").append(atn.getAckUser()).append("</td>");
         sbuf.append("<td>").append(atn.getAckSystem()).append("</td");
            
         sbuf.append("</tr>").append("\n");
      }
         
      sbuf.append("</table>").append("\n");
      
      return sbuf.toString(); 
   }
   
   // Lifecycle control (ServiceMBeanSupport) -----------------------
   
   /**
    * Start 
   **/
   public void startService()
      throws Exception
   {
      // subsbscribe myself for notifications
      super.subscribe(true);
   }
   
   /**
    * Stop
   **/
   public void stopService()
      throws Exception
   {
      // unsubscribe for notifications
      super.unsubscribe();
   }
   
   // ListenerServiceMBeanSupport -----------------------------------
   
   /**
    * Overriden to add handling!
   **/
   public void handleNotification2(Notification notification, Object handback)
   {
      log.info("Got notification (#" + Long.toString(this.notificationCount.increment())
             + "): " + notification + ", handback: " + handback);
      
      this.almtab.update(notification);
   }
}
