/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.test.webservice.attachment;

// $Id:AttachmentDIITestCase.java,v 1.0, 2005-06-24 18:46:10Z, Robert Worsnop$

import junit.framework.Test;
import org.jboss.test.JBossTestCase;

import javax.activation.DataHandler;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import javax.xml.namespace.QName;
import javax.xml.rpc.Call;
import javax.xml.rpc.ParameterMode;
import javax.xml.rpc.Service;
import javax.xml.rpc.ServiceFactory;
import java.awt.*;
import java.io.File;
import java.net.URL;
import java.rmi.RemoteException;

/**
 * Test SOAP with Attachements (SwA) through the JAXRPC DII layer.
 *
 * [ 1039881 ] Cannot create multipart/mixed SOAP attachment
 *
 * @author Thomas.Diesler@jboss.org
 * @since Nov 16, 2004
 */
public class AttachmentDIITestCase extends JBossTestCase
{
   private static final String NS_URI = "http://org.jboss.webservice/attachment";
   private static final QName SERVICE_NAME = new QName(NS_URI, "Attachment");
   private static final QName PORT_NAME = new QName(NS_URI, "AttachmentPort");

   private static final QName XSD_STRING = new QName("http://www.w3.org/2001/XMLSchema", "string");
   private static final QName AXIS_MIME_IMAGE = new QName("http://xml.apache.org/xml-soap", "Image");
   private static final QName AXIS_MIME_PLAINTEXT = new QName("http://xml.apache.org/xml-soap", "PlainText");
   private static final QName AXIS_MIME_MULTIPART = new QName("http://xml.apache.org/xml-soap", "Multipart");
   private static final QName AXIS_MIME_SOURCE = new QName("http://xml.apache.org/xml-soap", "Source");

   public AttachmentDIITestCase(String name)
   {
      super(name);
   }

   /** Send a multipart message with a text/plain attachment part
    */
   public void testSendMimeImageGIF() throws Exception
   {
      String rpcMethodName = "sendMimeImageGIF";
      Call call = setupMimeMessage(rpcMethodName, "image/gif");

      URL url = new File("resources/webservice/attachment/attach.gif").toURL();

      // On Linux the X11 server must be installed properly to create images successfully.
      // If the image cannot be created in the test VM, we assume it cannot be done on the
      // server either, so we just skip the test
      Image image = null;
      try
      {
         image = Toolkit.getDefaultToolkit().createImage(url);
      }
      catch (Throwable th)
      {
         log.warn("Cannot create Image: " + th);
      }

      if (image != null)
      {
         sendAndValidateMimeMessage(call, new DataHandler(url));
      }
   }

   /** Send a multipart message with a text/plain attachment part
    */
   public void testSendMimeImageJPEG() throws Exception
   {
      String rpcMethodName = "sendMimeImageJPEG";
      Call call = setupMimeMessage(rpcMethodName, "image/jpeg");

      URL url = new File("resources/webservice/attachment/attach.jpeg").toURL();

      // On Linux the X11 server must be installed properly to create images successfully.
      // If the image cannot be created in the test VM, we assume it cannot be done on the
      // server either, so we just skip the test
      Image image = null;
      try
      {
         image = Toolkit.getDefaultToolkit().createImage(url);
      }
      catch (Throwable th)
      {
         log.warn("Cannot create Image: " + th);
      }

      if (image != null)
      {
         sendAndValidateMimeMessage(call, new DataHandler(url));
      }
   }

   /** Send a multipart message with a text/plain attachment part
    */
   public void testSendMimeTextPlain() throws Exception
   {
      String rpcMethodName = "sendMimeTextPlain";
      Call call = setupMimeMessage(rpcMethodName, "text/plain");

      URL url = new File("resources/webservice/attachment/attach.txt").toURL();
      sendAndValidateMimeMessage(call, new DataHandler(url));
   }

   /** Send a multipart message with a text/plain attachment part
    */
   public void testSendMimeMultipart() throws Exception
   {
      String rpcMethodName = "sendMimeMultipart";
      Call call = setupMimeMessage(rpcMethodName, "multipart/mixed");

      URL url = new File("resources/webservice/attachment/attach.txt").toURL();
      MimeMultipart mimepart = new MimeMultipart("mixed");
      MimeBodyPart bodyPart = new MimeBodyPart();
      bodyPart.setDataHandler(new DataHandler(url));
      String bpct = bodyPart.getContentType();
      bodyPart.setHeader("Content-Type", bpct);
      mimepart.addBodyPart(bodyPart);

      sendAndValidateMimeMessage(call, mimepart);
   }

   /** Send a multipart message with a text/plain attachment part
    */
   public void testSendMimeTextXML() throws Exception
   {
      String rpcMethodName = "sendMimeTextXML";
      Call call = setupMimeMessage(rpcMethodName, "text/xml");

      URL url = new File("resources/webservice/attachment/attach.xml").toURL();
      sendAndValidateMimeMessage(call, new DataHandler(url));
   }

   /** Send a multipart message with a text/plain attachment part
    */
   public void testSendMimeApplicationXML() throws Exception
   {
      String rpcMethodName = "sendMimeApplicationXML";
      Call call = setupMimeMessage(rpcMethodName, "application/xml");

      URL url = new File("resources/webservice/attachment/attach.xml").toURL();
      sendAndValidateMimeMessage(call, new DataHandler(url));
   }

   /** Setup the multipart/related MIME message
    */
   private Call setupMimeMessage(String rpcMethodName, String contentType)
           throws Exception
   {
      ServiceFactory factory = ServiceFactory.newInstance();
      Service service = factory.createService(SERVICE_NAME);

      Call call = service.createCall(PORT_NAME, new QName(NS_URI, rpcMethodName));
      call.addParameter("message", XSD_STRING, ParameterMode.IN);

      if (contentType.equals("image/gif") || contentType.equals("image/jpeg"))
         call.addParameter("mimepart", AXIS_MIME_IMAGE, ParameterMode.IN);
      else if (contentType.equals("text/plain"))
         call.addParameter("mimepart", AXIS_MIME_PLAINTEXT, ParameterMode.IN);
      else if (contentType.startsWith("multipart/"))
         call.addParameter("mimepart", AXIS_MIME_MULTIPART, ParameterMode.IN);
      else if (contentType.equals("text/xml") || contentType.equals("application/xml"))
         call.addParameter("mimepart", AXIS_MIME_SOURCE, ParameterMode.IN);

      call.setReturnType(XSD_STRING);

      call.setTargetEndpointAddress("http://" + getServerHost() + ":8080/ws4ee-attachment");

      return call;
   }

   /** Send the message and validate the result
    */
   private void sendAndValidateMimeMessage(Call call, Object mimepart)
           throws RemoteException
   {
      String message = "Some text message";
      String value = (String)call.invoke(new Object[]{message, mimepart});

      System.out.println(value);
      assertEquals("[pass]", value);
   }

   /** Deploy the test ear */
   public static Test suite() throws Exception
   {
      return getDeploySetup(AttachmentDIITestCase.class, "ws4ee-attachment.war");
   }
}
