/*
 * JBoss, the OpenSource WebOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.test.security.ejb;

import javax.ejb.EJBException;
import javax.ejb.MessageDrivenBean;
import javax.ejb.MessageDrivenContext;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NameParser;
import javax.naming.NamingException;

import org.jboss.test.security.interfaces.ProjRepository;
import org.jboss.test.security.interfaces.ProjRepositoryHome;

/** An MDB that invokes several methods on the ProjRepository session bean,
 each of which require a seperate role to test the assignement of multiple
 roles to the run-as identity.
 
 @author Scott.Stark@jboss.org
 @version $Revision:1$
 */
public class RunAsWithRolesMDB implements MessageDrivenBean, MessageListener
{
   org.apache.log4j.Category log = org.apache.log4j.Category.getInstance(getClass());
   
   private MessageDrivenContext ctx = null;
   private InitialContext iniCtx;
   
   public RunAsWithRolesMDB()
   {
   }

   public void setMessageDrivenContext(MessageDrivenContext ctx)
      throws EJBException
   {
      this.ctx = ctx;
      try
      {
         iniCtx = new InitialContext();
      }
      catch(NamingException e)
      {
         throw new EJBException(e);
      }
   }
   
   public void ejbCreate()
   {
   }
   
   public void ejbRemove()
   {
      ctx = null;
   }

   public void onMessage(Message message)
   {
      Queue replyTo = null;
      try
      {
         replyTo = (Queue) message.getJMSReplyTo();
         String name = message.getStringProperty("name");
         ProjRepositoryHome home = (ProjRepositoryHome) iniCtx.lookup("java:comp/env/ejb/ProjRepository");
         NameParser parser = iniCtx.getNameParser("");
         Name projName = parser.parse(name);
         // This requires the ProjectAdmin role
         ProjRepository bean = home.create(projName);
         // This requires CreateFolder role
         Name programs = parser.parse("/Programs Files");
         bean.createFolder(programs);
         // This requires DeleteFolder role
         bean.deleteItem(programs);
         sendReply(replyTo, "Role tests ok");
         bean.remove();
      }
      catch(Throwable e)
      {
         log.debug("failed", e);
         if( replyTo != null )
            sendReply(replyTo, "Failed, ex="+e.getMessage());
      }
   }
   private void sendReply(Queue replyTo, String info)
   {
      try
      {
         InitialContext ctx = new InitialContext();
         QueueConnectionFactory queueFactory = (QueueConnectionFactory) ctx.lookup("java:comp/env/jms/QueFactory");
         QueueConnection queueConn = queueFactory.createQueueConnection();
         QueueSession session = queueConn.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
         Message msg = session.createMessage();
         msg.setStringProperty("reply", info);
         QueueSender sender = session.createSender(replyTo);
         sender.send(msg);
         sender.close();
         session.close();
         queueConn.close();
         log.info("Sent reply");
      }
      catch(Exception e)
      {
         log.error("Failed to send reply", e);
      }
   }
}
