/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.test.naming.test;


import java.net.URL;
import java.net.URLClassLoader;
import java.util.Properties;
import java.util.Hashtable;
import java.io.Serializable;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NamingException;
import javax.naming.Name;
import javax.naming.spi.StateFactory;
import javax.naming.spi.ObjectFactory;

import junit.framework.TestCase;
import junit.framework.Test;
import junit.framework.TestSuite;
import junit.textui.TestRunner;

import org.apache.log4j.Logger;

/** Simple unit tests for the jndi implementation.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:1$
 */
public class ImplUnitTestCase extends TestCase
{
   static final Logger log = Logger.getLogger(ImplUnitTestCase.class);

   
   /**
    * Constructor for the SimpleUnitTestCase object
    *
    * @param name  Test name
    */
   public ImplUnitTestCase(String name)
   {
      super(name);
   }

   /**
    * Tests that the second time you create a subcontext you get an exception.
    *
    * @exception Exception  Description of Exception
    */
   public void testCreateSubcontext() throws Exception
   {
      log.debug("+++ testCreateSubcontext");
      InitialContext ctx = getInitialContext();
      ctx.createSubcontext("foo");
      try
      {
         ctx.createSubcontext("foo");
         fail("Second createSubcontext(foo) did NOT fail");
      }
      catch (NameAlreadyBoundException e)
      {
         log.debug("Second createSubcontext(foo) failed as expected");
      }
      ctx.createSubcontext("foo/bar");
      ctx.unbind("foo/bar");
      ctx.unbind("foo");
   }

   /** Lookup a name to test basic connectivity and lookup of a known name
    *
    * @throws Exception
    */
   public void testLookup() throws Exception
   {
      log.debug("+++ testLookup");
      InitialContext ctx = getInitialContext();
      Object obj = ctx.lookup("");
      log.debug("lookup('') = "+obj);
   }

   public void testEncPerf() throws Exception
   {
      int count = Integer.getInteger("jbosstest.threadcount", 10).intValue();
      int iterations = Integer.getInteger("jbosstest.iterationcount", 1000).intValue();
      log.info("Creating "+count+"threads doing "+iterations+" iterations");
      InitialContext ctx = getInitialContext();
      URL[] empty = {};
      Thread[] testThreads = new Thread[count];
      for(int t = 0; t < count; t ++)
      {
         ClassLoader encLoader = URLClassLoader.newInstance(empty);
         Thread.currentThread().setContextClassLoader(encLoader);
         Runnable test = new ENCTester(ctx, iterations);
         Thread thr = new Thread(test, "Tester#"+t);
         thr.setContextClassLoader(encLoader);
         thr.start();
         testThreads[t] = thr;
      }

      for(int t = 0; t < count; t ++)
      {
         Thread thr = testThreads[t];
         thr.join();
      }
   }

   /**
    * 
    * @throws NamingException
    */ 
   public void testFactorySupport() throws NamingException
   {
      log.info("+++ testFactorySupport");
      NotSerializableObject nso = new NotSerializableObject( "nsc" );
      Context ctx = getInitialContext();

      try 
      {
         ctx.bind("test", nso);
         fail();
      }
      catch( NamingException ex )
      {
         log.debug("bind failed as expected", ex);
      }

      Properties env = new Properties();
      env.setProperty(Context.STATE_FACTORIES, TestFactory.class.getName());
      env.setProperty(Context.OBJECT_FACTORIES, TestFactory.class.getName());
      ctx = new InitialContext(env);

      log.debug("Retest with TestFactory enabled");
      ctx.bind("test", nso);

      Object boundObject = ctx.lookup( "test" );
      assertNotNull( boundObject );
      // make sure it's of type NotSerializableObject
      NotSerializableObject nso2 = (NotSerializableObject) boundObject;
      assertEquals( nso.getId(), nso2.getId() );
   }

   static InitialContext getInitialContext() throws NamingException
   {
      InitialContext ctx = new InitialContext();
      return ctx;
   }

   private static class ENCTester implements Runnable
   {
      Context enc;
      int iterations;

      ENCTester(InitialContext ctx, int iterations) throws Exception
      {
         log.info("CL: "+Thread.currentThread().getContextClassLoader());
         this.iterations = iterations;
         enc = (Context) ctx.lookup("java:comp");
         enc = enc.createSubcontext("env");
         enc.bind("int", new Integer(1));
         enc.bind("double", new Double(1.234));
         enc.bind("string", "str");
         enc.bind("url", new URL("http://www.jboss.org"));
      }

      public void run()
      {
         try
         {
            InitialContext ctx =  new InitialContext();
            for(int i = 0; i < iterations; i ++)
            {
               Integer i1 = (Integer) enc.lookup("int");
               log.debug("int: "+i1);
               i1 = (Integer) ctx.lookup("java:comp/env/int");
               log.debug("java:comp/env/int: "+i1);
               Double d = (Double) enc.lookup("double");
               log.debug("double: "+d);
               d = (Double) ctx.lookup("java:comp/env/double");
               log.debug("java:comp/env/double: "+d);
               String s = (String) enc.lookup("string");
               log.debug("string: "+s);
               s = (String) ctx.lookup("java:comp/env/string");
               log.debug("java:comp/env/string: "+s);
               URL u = (URL) enc.lookup("url");
               log.debug("url: "+u);
               u = (URL) ctx.lookup("java:comp/env/url");
               log.debug("java:comp/env/url: "+u);
            }
         }
         catch(Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   private static class NotSerializableObject
   {
      protected String id;
      
      public NotSerializableObject() {}
      
      public NotSerializableObject( String id ) 
      {
         this.id = id;
      }
      
      public String getId() 
      {
         return id;
      }
      
      public String toString()
      {
         return "NotSerializableObject<" + getId() + ">";
      }
   }
   
   private static class SerializableObject extends NotSerializableObject
      implements Serializable
   {
      private static final long serialVersionUID = 1;

      public SerializableObject () {}
      
      public SerializableObject (String id)
      {
         super( id );
      }
      
      public String toString()
      {
         return "SerializableObject<" + getId() + ">";
      }
      
      private void writeObject(ObjectOutputStream out)
         throws IOException
      {
         out.writeObject(getId());
      }
      private void readObject(ObjectInputStream in)
         throws IOException, ClassNotFoundException
      {
         id = (String) in.readObject();
      }
      
   }
   
   
   public static class TestFactory implements StateFactory, ObjectFactory
   {
      public Object getStateToBind (Object obj, Name name, Context nameCtx,
         Hashtable environment) throws NamingException
      {
         if( obj instanceof NotSerializableObject )
         {
            String id = ((NotSerializableObject) obj).getId();
            return new SerializableObject( id );
         }
         
         return null;
      }

      public Object getObjectInstance (Object obj, Name name, Context nameCtx,
         Hashtable env) throws Exception
      {
         log.debug("getObjectInstance, obj:" + obj + ", name: " + name
            + ", nameCtx: " + nameCtx +", env: "+env);
         if( obj instanceof SerializableObject )
         {
            String id = ((SerializableObject) obj).getId();
            return new NotSerializableObject( id );
         }
         
         return null;
      }
   }

   public static Test suite()
   {
      TestSuite suite = new TestSuite();
      suite.addTest(new TestSuite(ImplUnitTestCase.class));

      // Create an initializer for the test suite
      NamingServerSetup wrapper = new NamingServerSetup(suite);
      return wrapper; 
   }

   /** Used to run the testcase from the command line
    *
    * @param args  The command line arguments
    */
   public static void main(String[] args)
   {
      TestRunner.run(ImplUnitTestCase.suite());
   }
}
