/*
 *
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.test.cache.perf.aop;


import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.PropertyConfigurator;
import org.jboss.cache.TreeCache;
import org.jboss.cache.aop.TreeCacheAop;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.lock.LockStrategyFactory;
import org.jboss.cache.transaction.DummyTransactionManager;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.transaction.UserTransaction;
import java.text.DecimalFormat;
import java.text.FieldPosition;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * Local mode performance test for TreeCache.
 *
 * @version $Revision:1$
 * @author<a href="mailto:bwang@jboss.org">Ben Wang</a> May 20 2003
 */
public class ReplicatedAsyncMapPerfAopTest extends TestCase
{
   TreeCacheAop cache_;
   int cachingMode_ = TreeCache.LOCAL;
   final static Properties p_;
//   final static Logger log_ = Logger.getLogger(LocalPerfAopTest.class);
   String oldFactory_ = null;
   final String FACTORY = "org.jboss.cache.transaction.DummyContextFactory";
   DummyTransactionManager tm_;
   Map[] proxyMaps_ = null;
   ArrayList nodeList_;
   static final int depth_ = 3;
   static final int children_ = 4;
   static final int mapValueSize_ = 100;
   static final String seed1_ = "This is a test. ";
   static final String seed2_ = "THAT is a TEST. ";
   StringBuffer originalStrBuf_;
   StringBuffer newStrBuf_;

   static
   {
      p_ = new Properties();
      p_.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.cache.transaction.DummyContextFactory");
   }

   public ReplicatedAsyncMapPerfAopTest(String name)
   {
      super(name);
   }

   public void setUp() throws Exception
   {
      super.setUp();

      oldFactory_ = System.getProperty(Context.INITIAL_CONTEXT_FACTORY);
      System.setProperty(Context.INITIAL_CONTEXT_FACTORY, FACTORY);

      DummyTransactionManager.getInstance();
      initCaches(TreeCache.LOCAL);
      tm_ = new DummyTransactionManager();

      originalStrBuf_ = new StringBuffer();
      newStrBuf_ = new StringBuffer();
      generateString();
      log("ReplicatedAsyncPerfAopTest: cacheMode=ReplAsync");
      nodeList_ = nodeGen(depth_, children_);
      populateNode();
   }

   private void generateString()
   {
      int length = seed1_.length();
      boolean isTrue = false;
      while (originalStrBuf_.length() < mapValueSize_) {
         originalStrBuf_.append(seed1_);
         newStrBuf_.append(seed2_);
      }
   }

   private void populateNode() throws Exception
   {
      // Formating
      DecimalFormat form = new DecimalFormat("#.00");
      FieldPosition fieldPos = new FieldPosition(0);
      StringBuffer dumbStr = new StringBuffer();
      proxyMaps_ = new Map[nodeList_.size()];

      long time1 = System.currentTimeMillis();
      int nOps = 0;
      for (int i = 0; i < nodeList_.size(); i++) {
//            String key = Integer.toString(i);
         // put the cache in the aop first
         Map map = populateMap();
         cache_.putObject((String) nodeList_.get(i), map);
         proxyMaps_[i] = (Map) cache_.getObject((String) nodeList_.get(i));
      }

      nOps = nodeList_.size();
      long time2 = System.currentTimeMillis();
      double d = (double) (time2 - time1) / nOps;
      log("Time elapsed for one putObject and getObject entry is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
   }

   private Map populateMap()
   {
      Map map = new HashMap();
      for (int i = 0; i < nodeList_.size(); i++) {
         String key = Integer.toString(i) + "aop";
         String value = originalStrBuf_.toString();
         map.put(key, value);
      }
      return map;
   }

   public void tearDown() throws Exception
   {
      super.tearDown();

      DummyTransactionManager.destroy();
      destroyCaches();

      if (oldFactory_ != null) {
         System.setProperty(Context.INITIAL_CONTEXT_FACTORY, oldFactory_);
         oldFactory_ = null;
      }

      proxyMaps_ = null;
   }

   void initCaches(int caching_mode) throws Exception
   {
      cachingMode_ = caching_mode;
      cache_ = new TreeCacheAop();
      PropertyConfigurator config = new PropertyConfigurator();
      config.configure(cache_, "META-INF/replAsync-service.xml"); // read in generic replAsync xml
      cache_.setTransactionManagerLookupClass("org.jboss.cache.DummyTransactionManagerLookup");
      cache_.startService();
//        org.jgroups.log.Trace.init();
   }

   void destroyCaches() throws Exception
   {
      cache_.stopService();
      cache_ = null;
   }

   public void testAll() throws Exception
   {

      log("=== Start ===");
//        try { Thread.sleep(5000); } catch (Exception ex) {};

      // Formating
      DecimalFormat form = new DecimalFormat("#.00");
      FieldPosition fieldPos = new FieldPosition(0);
      StringBuffer dumbStr = new StringBuffer();
      boolean hasTx = false;

      // Step 1. Add entries to the cache
      long time1 = System.currentTimeMillis();
      int nOps = 0;
      nOps = _put(hasTx);
      long time2 = System.currentTimeMillis();
      double d = (double) (time2 - time1) / nOps;
      log("Time elapsed for put entry is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 2. Query the cache
      time1 = System.currentTimeMillis();
      nOps = _get(hasTx);
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for get entry is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 3. Remove entries from the cache
      time1 = System.currentTimeMillis();
      nOps = _remove(hasTx);
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for remove entry is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      log("=== End ===");
   }

   protected void setLevelRW()
   {
      log("set lock level to RWUpgrade ...");
      LockStrategyFactory.setIsolationLevel(IsolationLevel.REPEATABLE_READ);
   }

   protected void setLevelSerial()
   {
      log("set lock level to SimpleLock ...");
      LockStrategyFactory.setIsolationLevel(IsolationLevel.SERIALIZABLE);
   }

   private int _put(boolean hasTx) throws Exception
   {
      UserTransaction tx = null;
      if (hasTx) {
         tx = (UserTransaction) new InitialContext(p_).lookup("UserTransaction");
      }

      String value = newStrBuf_.toString();
      for (int i = 0; i < nodeList_.size(); i++) {
         String key = Integer.toString(i) + "aop";
         if (hasTx) {
            tx.begin();
            proxyMaps_[i].put(key, value);
            tx.commit();
         } else {
            proxyMaps_[i].put(key, value);
         }
      }

      return nodeList_.size();
   }

   private int _get(boolean hasTx) throws Exception
   {
      UserTransaction tx = null;
      if (hasTx) {
         tx = (UserTransaction) new InitialContext(p_).lookup("UserTransaction");
      }

      for (int i = 1; i < nodeList_.size(); i++) {
         String key = Integer.toString(i) + "aop";
         if (hasTx) {
            tx.begin();
            String str = (String) proxyMaps_[i].get(key);
//                log("_get(): key: " + key + " value: " +str);
            tx.commit();
         } else {
            String str = (String) proxyMaps_[i].get(key);
//                log("_get(): key: " + key + ", value: " +str);
         }
      }

      return nodeList_.size();
   }

   private int _remove(boolean hasTx) throws Exception
   {
      UserTransaction tx = null;
      if (hasTx) {
         tx = (UserTransaction) new InitialContext(p_).lookup("UserTransaction");
      }

      for (int i = 1; i < nodeList_.size(); i++) {
         String key = Integer.toString(i) + "aop";
         if (hasTx) {
            tx.begin();
            proxyMaps_[i].remove(key);
            tx.commit();
         } else {
            proxyMaps_[i].remove(key);
         }
      }

      return nodeList_.size();
   }

   /**
    * Generate the tree nodes quasi-exponentially. I.e., depth is the level
    * of the hierarchy and children is the number of children under each node.
    * This strucutre is used to add, get, and remove for each node.
    */
   private ArrayList nodeGen(int depth, int children)
   {
      ArrayList strList = new ArrayList();
      ArrayList oldList = new ArrayList();
      ArrayList newList = new ArrayList();

      oldList.add("/");
      newList.add("/");
      strList.add("/");

      while (depth > 0) {
         // Trying to produce node name at this depth.
         newList = new ArrayList();
         for (int i = 0; i < oldList.size(); i++) {
            for (int j = 0; j < children; j++) {
               String tmp = (String) oldList.get(i);
               tmp += Integer.toString(j);
               if (depth != 1) tmp += "/";
               newList.add(tmp);
            }
         }
         strList.addAll(newList);
         oldList = newList;
         depth--;
      }

      log("Nodes generated: " + strList.size());
      return strList;
   }

   public static Test suite() throws Exception
   {
      return new TestSuite(ReplicatedAsyncMapPerfAopTest.class);
   }

   private void log(String str)
   {
//     System.out.println(this.getClass().getName() +": " +str);
      System.out.println(str);
   }

}
