/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.security.plugins;

import java.security.PrivilegedAction;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.security.PrivilegedActionException;
import java.security.Principal;
import java.util.Set;
import java.util.Iterator;
import javax.security.auth.Subject;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;
import javax.security.auth.callback.CallbackHandler;

import org.jboss.security.SecurityAssociation;

/** Common PrivilegedAction used by classes in this package.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revison:$
 */
class SubjectActions
{
   private static class GetSubjectAction implements PrivilegedAction
   {
      static PrivilegedAction ACTION = new GetSubjectAction();
      public Object run()
      {
         Subject subject = SecurityAssociation.getSubject();
         return subject;
      }
   }

   private static class CopySubjectAction implements PrivilegedAction
   {
      Subject fromSubject;
      Subject toSubject;
      boolean setReadOnly;
      CopySubjectAction(Subject fromSubject, Subject toSubject, boolean setReadOnly)
      {
         this.fromSubject = fromSubject;
         this.toSubject = toSubject;
         this.setReadOnly = setReadOnly;
      }
      public Object run()
      {
         Set principals = fromSubject.getPrincipals();
         Set principals2 = toSubject.getPrincipals();
         Iterator iter = principals.iterator();
         while( iter.hasNext() )
            principals2.add(iter.next());
         Set privateCreds = fromSubject.getPrivateCredentials();
         Set privateCreds2 = toSubject.getPrivateCredentials();
         iter = privateCreds.iterator();
         while( iter.hasNext() )
            privateCreds2.add(iter.next());
         Set publicCreds = fromSubject.getPublicCredentials();
         Set publicCreds2 = toSubject.getPublicCredentials();
         iter = publicCreds.iterator();
         while( iter.hasNext() )
            publicCreds2.add(iter.next());
         if( setReadOnly == true )
            toSubject.setReadOnly();
         return null;
      }
   }

   private static class LoginContextAction implements PrivilegedExceptionAction
   {
      String securityDomain;
      Subject subject;
      CallbackHandler handler;
      LoginContextAction(String securityDomain, Subject subject,
         CallbackHandler handler)
      {
         this.securityDomain = securityDomain;
         this.subject = subject;
         this.handler = handler;
      }
      public Object run() throws Exception
      {
         LoginContext lc = new LoginContext(securityDomain, subject, handler);
         return lc;
      }
   }

   private static class GetTCLAction implements PrivilegedAction
   {
      static PrivilegedAction ACTION = new GetTCLAction();
      public Object run()
      {
         ClassLoader loader = Thread.currentThread().getContextClassLoader();
         return loader;
      }
   }

   private static class SetContextInfoAction implements PrivilegedAction
   {
      Object key;
      Object value;
      SetContextInfoAction(Object key, Object value)
      {
         this.key = key;
         this.value = value;
      }
      public Object run()
      {
         return SecurityAssociation.setContextInfo(key, value);
      }
   }

   interface PrincipalInfoAction
   {
      PrincipalInfoAction PRIVILEGED = new PrincipalInfoAction()
      {
         public void push(final Principal principal, final Object credential,
            final Subject subject)
         {
            AccessController.doPrivileged(
               new PrivilegedAction()
               {
                  public Object run()
                  {
                     SecurityAssociation.pushSubjectContext(subject, principal, credential);
                     return null;
                  }
               }
            );
         }
         public void pop()
         {
            AccessController.doPrivileged(
               new PrivilegedAction()
               {
                  public Object run()
                  {
                     SecurityAssociation.popSubjectContext();
                     return null;
                  }
               }
            );
         }
      };

      PrincipalInfoAction NON_PRIVILEGED = new PrincipalInfoAction()
      {
         public void push(Principal principal, Object credential, Subject subject)
         {
            SecurityAssociation.pushSubjectContext(subject, principal, credential);
         }
         public void pop()
         {
            SecurityAssociation.popSubjectContext();
         }
      };

      void push(Principal principal, Object credential, Subject subject);
      void pop();
   }

   static Subject getActiveSubject()
   {
      Subject subject = (Subject) AccessController.doPrivileged(GetSubjectAction.ACTION);
      return subject;
   }
   static void copySubject(Subject fromSubject, Subject toSubject)
   {
      copySubject(fromSubject, toSubject, false);
   }
   static void copySubject(Subject fromSubject, Subject toSubject, boolean setReadOnly)
   {
      CopySubjectAction action = new CopySubjectAction(fromSubject, toSubject, setReadOnly);
      AccessController.doPrivileged(action);
   }

   static LoginContext createLoginContext(String securityDomain, Subject subject,
      CallbackHandler handler)
      throws LoginException
   {
      LoginContextAction action = new LoginContextAction(securityDomain, subject, handler);
      try
      {
         LoginContext lc = (LoginContext) AccessController.doPrivileged(action);
         return lc;
      }
      catch(PrivilegedActionException e)
      {
         Exception ex = e.getException();
         if( ex instanceof LoginException )
            throw (LoginException) ex;
         else
            throw new LoginException(ex.getMessage());
      }
   }

   static ClassLoader getContextClassLoader()
   {
      ClassLoader loader = (ClassLoader) AccessController.doPrivileged(GetTCLAction.ACTION);
      return loader;
   }

   static Object setContextInfo(Object key, Object value)
   {
      SetContextInfoAction action = new SetContextInfoAction(key, value);
      Object prevInfo = AccessController.doPrivileged(action);
      return prevInfo;
   }

   static void pushSubjectContext(Principal principal, Object credential,
      Subject subject)
   {
      if(System.getSecurityManager() == null)
      {
         PrincipalInfoAction.NON_PRIVILEGED.push(principal, credential, subject);
      }
      else
      {
         PrincipalInfoAction.PRIVILEGED.push(principal, credential, subject);
      }
   }
   static void popSubjectContext()
   {
      if(System.getSecurityManager() == null)
      {
         PrincipalInfoAction.NON_PRIVILEGED.pop();
      }
      else
      {
         PrincipalInfoAction.PRIVILEGED.pop();
      }
   }

}

